#!/usr/bin/env jython

# ===========================================
# 脚本功能概览
# 1) 加载 UI/语言/参数配置
# 2) 读取 OpenPnP 当前工具与 Z 值并完成计算
# 3) 读取或写回 USB 的 config.txt（支持手动选择）
# 4) 展示结果与历史记录，给出操作提示
# ===========================================
# 注意：运行于 OpenPnP 的 Jython 脚本环境
# ===========================================

# Z轴平衡计算器（Swing UI，源自 Calculator.HTML） / Z-Axis Balancing Calculator (Swing UI, from Calculator.HTML)
# 在 OpenPnP 的 Jython 脚本环境中运行 / Run inside OpenPnP's Jython scripting environment

from javax.swing import (
    JFrame, JPanel, JLabel, JTextField, JButton, JTable, JScrollPane,
    BoxLayout, JOptionPane, JTextPane, SwingUtilities, BorderFactory, JFileChooser, Box, Timer
)
from javax.swing.table import DefaultTableModel, DefaultTableCellRenderer
from java.awt import Dimension, Font, Color, BorderLayout
import java.lang

import os, io, re, math
from java.nio.file import Paths, Files
from java.io import File, FileInputStream, InputStreamReader
from java.nio.charset import StandardCharsets
from java.util import Properties
from org.openpnp.gui import MainFrame
try:
    from org.openpnp.model import Configuration
    from org.openpnp.util import UiUtils, MovableUtils
except Exception:
    Configuration = None
    UiUtils = None
    MovableUtils = None


# ===========================================
# 配置区：UI、USB 与多语言相关常量
# ===========================================
_USB_FIND_STATUS = None
_USB_SELECTED_ROOT = None


class UIConfig(object):
    # 计算阈值（mm） / Threshold (mm)
    THRESHOLD = 0.1
    # 浮点比较容差 / Float compare tolerance
    FLOAT_TOLERANCE = 1e-10
    # 历史记录条数 / History record count
    MAX_HISTORY = 5
    # 调整系数 / Adjustment factor
    ADJUST_FACTOR = 2.5

    # USB 配置变量名 / USB config variable name
    VAR_NAME = "endstop.maxz.homing_position"
    # 自动 USB 查找延迟（秒） / Auto USB lookup delay (seconds)
    AUTO_USB_FIND_DELAY_SECONDS = 1.5

    # UI 尺寸参数（像素） / UI size parameters (px)
    INPUT_FIELD_WIDTH = 240  # 输入框宽度 / input field width
    LABEL_WIDTH = 150        # 标签宽度 / label width
    BUTTON_WIDTH = 150       # 按钮宽度 / button width
    H_GAP = 8                # 横向间距 / horizontal gap
    CONTENT_WIDTH = 480      # 内容区宽度 / content width
    FRAME_WIDTH_PAD = 140    # 窗口宽度补偿 / frame width padding
    FRAME_HEIGHT = 600       # 窗口高度 / frame height
    TITLE_HEIGHT = 32        # 标题高度 / title height
    TITLE_GAP = 12           # 标题与输入面板间距 / gap below title
    INPUT_PANEL_HEIGHT = 120 # 输入面板高度 / input panel height
    ROW_HEIGHT = 26          # 输入行高度 / input row height
    INPUT_ROW_GAP = 6        # 输入行间距 / input row gap
    BUTTON_PANEL_HEIGHT = 48 # 按钮面板高度 / button panel height
    BUTTON_HEIGHT = 28       # 按钮高度 / button height
    RESULT_PANEL_HEIGHT = 200 # 结果面板高度 / result panel height
    RESULT_AREA_HEIGHT = 180  # 结果区域高度 / result area height
    HISTORY_PANEL_HEIGHT = 200 # 历史面板高度 / history panel height
    HISTORY_SCROLL_HEIGHT = 180 # 历史表格高度 / history table height
    HISTORY_COL_UNIT_DIVISOR = 5.5 # 列宽比例系数 / column width divisor
    BUTTON_GAP = 8           # 按钮之间间距 / gap between buttons
    LANG_BUTTON_WIDTH = 90   # 语言按钮宽度 / language button width

    # 标题样式 / Title style
    TITLE_FONT_SIZE = 20     # 标题字体大小 / title font size
    TITLE_COLOR = Color(0, 0, 0)  # 标题颜色 / title color
    TITLE_FONT_NAME = None  # 标题字体（自动选择）/ title font name (auto)
    FONT_FALLBACKS = [  # 字体回退列表 / font fallback list
        "Microsoft YaHei",
        "PingFang SC",
        "Hiragino Sans GB",
        "Noto Sans CJK SC",
        "WenQuanYi Micro Hei",
        "Dialog",
    ]
    LABEL_FONT_SIZE = 12     # 输入标签字号 / label font size
    INPUT_FONT_SIZE = 12     # 输入框字号 / input field font size

    # 语言配置 / Language config
    DEFAULT_LANG = "zh"      # 默认语言 / default language
    PROPS_ZH = "CAL_zh.properties"  # 中文词条 / Chinese properties
    PROPS_EN = "CAL_en.properties"  # 英文词条 / English properties
    LANG_LABEL_ZH = "lang.zh"   # 中文按钮 key / Chinese button key
    LANG_LABEL_EN = "lang.en"  # 英文按钮 key / English button key

    # 工具名称前缀 / Tool name prefixes
    N1_PREFIX = "N1"  # N1 选择项前缀 / N1 selection prefix
    N2_PREFIX = "N2"  # N2 选择项前缀 / N2 selection prefix

    # 提示文本 / Message texts
    MSG_NOT_N1 = "Current tool is not N1.Please check the tool in Machine Controls!!"
    MSG_NOT_N2 = "Current tool is not N2.Please check the tool in Machine Controls!!"
    MSG_NEED_NEG_N1 = "N1 height should be negative value"
    MSG_NEED_NEG_N2 = "N2 height should be negative value"
    MSG_NO_TOOL = "no tool selected, need tool check!!"
    MSG_NO_PANEL = "No tool selected.Please check the tool in Machine Controls!!"

    # 按钮颜色（可选）/ Button colors (optional)
    CALC_BUTTON_COLOR = None   # 计算按钮颜色 / calculate button color
    CLEAR_BUTTON_COLOR = None  # 清除按钮颜色 / clear button color
    REWRITE_BUTTON_COLOR = None # 重写按钮颜色 / rewrite button color
    REWRITE_BREATH_COLOR = (0, 176, 80)  # 呼吸颜色 / breathing color
    REWRITE_BREATH_INTERVAL_MS = 50      # 呼吸刷新间隔 / breath tick ms
    REWRITE_BREATH_PERIOD_MS = 1200      # 呼吸周期 / breath period ms



# ---------------------- 多语言文本 / Inline i18n ----------------------
# ===========================================
# 内置多语言文本：界面/提示/对话框文案
# ===========================================
INLINE_I18N_ZH = {
    "lang.zh": u"中文",
    "lang.en": u"English",
    "title.label": u"Z轴配平计算器",
    "title.frame": u"Z轴配平计算器--AlanDesign",
    "panel.inputs": u"数据 - 均为负数值",
    "panel.result": u"结果",
    "panel.history": u"最近5条记录",
    "label.balance": u"当前配平值:",
    "label.n1": u"N1 高度 :",
    "label.n2": u"N2 高度 :",
    "button.read_balance": u"读取配平值",
    "button.read_n1": u"读取 N1 高度",
    "button.read_n2": u"读取 N2 高度",
    "button.calculate": u"计算",
    "button.clear": u"清空",
    "button.rewrite": u"重写配平值",
    "col.no": u"序号",
    "col.balance": u"原配平值",
    "col.n1": u"N1 高度",
    "col.n2": u"N2 高度",
    "col.new_balance": u"新配平值",
    "col.status": u"状态",
    "guide.title": u"操作向导",
    "guide.step1": u"步骤<b> 1:</b> 点击 <b>读取配平值</b> 按钮，从U盘的config.txt中读取当前配平值。",
    "guide.step2": u"步骤<b> 2:</b> 依次手动测量并读取 <b>N1的高度</b> 与 <b>N2的高度</b>。(N1在左，N2在右)",
    "guide.step3": u"步骤<b> 3:</b> 点击 <b>计算</b> 按钮，获得新的配平值。",
    "guide.step4": u"步骤<b> 4:</b> 点击 <b>重写配平值</b> 按钮，即可将新的配平值写入U盘中的config.txt。",
    "guide.step6": u"<html><span style='font-size:14pt;'>点击 <b>重写配平值</b> 按钮，即可将新的配平值写入U盘的config.txt中。<br>同时，贴装头会自动收回下降中的吸嘴，并自动前往停靠点。</html>",
    "result.balanced": u"已配平",
    "result.pending": u"需要配平",
    "result.conclusion": u"高度差小于 0.1mm，N1&N2已完成配平！！",
    "result.new_balance": u"新配平值:",
    "msg.no_usb": u"未检测到有效的U盘和config.txt文件，\n请检查。",
    "msg.var_not_found": u"未在 {path} 中找到变量 {var}",
    "msg.parse_failed": u"无法获得 config.txt 中的配平值",
    "msg.no_tool": u"未选择控件，\n请检查当前机器控件中的所选项！",
    "msg.not_n1": u"当前控件不是 N1\n请检查当前机器控件，是否为 N1！",
    "msg.not_n2": u"当前控件不是 N2\n请检查当前机器控件，是否为 N2！",
    "msg.need_neg_n1": u"请检查 N1 的高度，应为负数值",
    "msg.need_neg_n2": u"请检查 N2 的高度，应为负数值",
    "msg.no_panel": u"机器控件不可用",
    "msg.no_balance_to_write": u"没有可写入的新配平值，请先计算。",
    "msg.write_ok": u"<html>新配平值已成功写入U盘的config.txt中。<br><br><span style='color:#D00000; font-size:16pt;'>接下来，还需要再进行一轮新的测量和计算。</span><br><br>请按照以下步骤完成后续操作：<span style='font-size:16pt;'><br>①务必，按压贴片机的复位按钮，持续3秒后松手；<br>②点击下方OK按钮后，前往操作面板；<br>③点击黑色小房子，重新执行归位；<br>④根据操作向导，重新开始新一轮的测量和计算。</span></html>",
    "msg.write_fail": u"写入失败：{err}",
    "dialog.usb": u"U盘",
    "dialog.input_error": u"输入错误",
    "dialog.rewrite": u"重新配平值-已完成",
    "error.required": u"{label} 为必填项。",
    "error.number": u"{label} 必须是数字。",
    "error.nonpositive": u"{label} 必须小于或等于 0。",
    "dialog.select_config.title": u"请选择 config.txt 文件",
    "msg.usb_config_not_found": u"未找到U盘中的 config.txt 文件",
}

INLINE_I18N_EN = {
    "lang.zh": u"中文",
    "lang.en": u"English",
    "title.label": u"Z-Axis Balancing Calculator",
    "title.frame": u"Z-Axis Balancing Calculator--by AlanDesign",
    "panel.inputs": u"Inputs - Negative values required",
    "panel.result": u"Result",
    "panel.history": u"Last 5 Records",
    "label.balance": u"Current Balance:",
    "label.n1": u"N1 Height :",
    "label.n2": u"N2 Height :",
    "button.read_balance": u"Read Balance",
    "button.read_n1": u"Read N1",
    "button.read_n2": u"Read N2",
    "button.calculate": u"Calculate",
    "button.clear": u"Clear",
    "button.rewrite": u"Rewrite Balance",
    "col.no": u"No.",
    "col.balance": u"Old Balance",
    "col.n1": u"N1 Height",
    "col.n2": u"N2 Height",
    "col.new_balance": u"New Balance",
    "col.status": u"Status",
    "guide.title": u"User Guide",
    "guide.step1": u"<b>Step 1:</b> Click <b>Read Balance</b> to read the balance value from USB config.txt.",
    "guide.step2": u"<b>Step 2:</b> Manually measure and read <b>N1 height</b> and <b>N2 height</b> in sequence.(N1 is on the left, N2 is on the right)",
    "guide.step3": u"<b>Step 3:</b> Click <b>Calculate</b> to get the new balance value.",
    "guide.step4": u"<b>Step 4:</b> Click <b>Rewrite Balance</b> to write the balance value back into the USB config.txt.",
    "guide.step6": u"<html><span style='font-size:14pt;'>Click the <b>Rewrite Balance</b> button to write the new balance values into the USB config.txt.<br>At the same time, the Head will automatically retract the descending nozzletip and move to the Park Location automatically.</html>",
    "result.balanced": u"Balanced",
    "result.pending": u"Adjust Needed",
    "result.conclusion": u"Height difference below 0.1mm, N1&N2 Balanced !<br> Well Done !!",
    "result.new_balance": u"New Balance value:",
    "msg.no_usb": u"No valid USB drive or config.txt file detected. \nPlease check.",
    "msg.var_not_found": u"Variable {var} not found in {path}",
    "msg.parse_failed": u"Failed to parse balance value from config.txt.",
    "msg.no_tool": u"No tool selected\nPlease check the tool in Machine Controls!!",
    "msg.not_n1": u"Current tool is not N1\nPlease check the tool in Machine Controls!!",
    "msg.not_n2": u"Current tool is not N2\nPlease check the tool in Machine Controls!!",
    "msg.need_neg_n1": u"N1 height should be negative value",
    "msg.need_neg_n2": u"N2 height should be negative value",
    "msg.no_panel": u"Machine Controls not available.",
    "msg.no_balance_to_write": u"No new balance to write. Calculate first and ensure Adjust Needed.",
    "msg.write_ok": u"<html>The new balance value has been successfully written to the USB config.txt.<br><br><span style='color:#D00000; font-size:16pt;'>Next, a new round of measurement and calculation is still required.</span><br><br>Please follow the steps below to complete the subsequent operations:<br><span style='font-size:16pt;'><br>① Be sure to press and hold the Reset Button nearby the Power Switch for 3 seconds, then release;<br>② After clicking the OK button below, go to the Jog panel;<br>③ Click the Black Home icon to re-execute the homing operation;<br>④ Follow the Operation Guide to start a new round of measurement and calculation.</span></html>",
    "msg.write_fail": u"Failed to write config: {err}",
    "dialog.usb": u"USB driver",
    "dialog.input_error": u"Input Error",
    "dialog.rewrite": u"Rewrite Balance",
    "error.required": u"{label} is required.",
    "error.number": u"{label} must be a number.",
    "error.nonpositive": u"{label} must be <= 0.",
    "dialog.select_config.title": u"Please select the config.txt file",
    "msg.usb_config_not_found": u"USB config not found.",
}

# ===========================================
# 资源路径与属性加载
# ===========================================
def _get_scripts_dir():
    try:
        scripts_root = scripting.getScriptsDirectory().toString()
        tornado_dir = File(scripts_root, "TornadoSMT")
        if tornado_dir.exists():
            return tornado_dir.getAbsolutePath()
        return scripts_root
    except Exception:
        pass
    try:
        from org.openpnp.model import Configuration
        cfg = Configuration.get()
        config_dir = cfg.getConfigurationDirectory()
        scripts_root = File(config_dir, "scripts")
        tornado_dir = File(scripts_root, "TornadoSMT")
        if tornado_dir.exists():
            return tornado_dir.getAbsolutePath()
        if scripts_root.exists():
            return scripts_root.getAbsolutePath()
    except Exception:
        try:
            here = os.path.dirname(__file__)
            return here
        except Exception:
            return os.getcwd()


def _load_properties_file(path):
    data = {}
    if path is None:
        return data
    try:
        f = File(path)
        if not f.exists():
            return data
        props = Properties()
        stream = FileInputStream(f)
        reader = InputStreamReader(stream, StandardCharsets.UTF_8)
        try:
            props.load(reader)
        finally:
            try:
                reader.close()
            except Exception:
                pass
            try:
                stream.close()
            except Exception:
                pass
        it = props.entrySet().iterator()
        try:
            bom = unichr(0xfeff)
        except Exception:
            bom = None
        while it.hasNext():
            entry = it.next()
            try:
                key = entry.getKey()
                val = entry.getValue()
            except Exception:
                continue
            try:
                key = str(key)
            except Exception:
                continue
            if bom and key.startswith(bom):
                key = key[len(bom):]
            try:
                data[key] = str(val)
            except Exception:
                data[key] = val
    except Exception:
        pass
    return data


def _load_translations(lang):
    if lang == "en":
        return INLINE_I18N_EN
    return INLINE_I18N_ZH
def _pick_font_family(fallbacks):
    sample = u"中文ABC"
    for name in fallbacks:
        try:
            f = Font(name, Font.PLAIN, 12)
            try:
                if f.canDisplayUpTo(sample) == -1:
                    return name
            except Exception:
                try:
                    if f.getFamily() == name:
                        return name
                except Exception:
                    pass
        except Exception:
            pass
    return "Dialog"


def _to_color(value, fallback=None):
    if value is None:
        return fallback
    try:
        if isinstance(value, Color):
            return value
    except Exception:
        pass
    try:
        if isinstance(value, tuple) and len(value) == 3:
            return Color(value[0], value[1], value[2])
    except Exception:
        pass
    return fallback


# ===========================================
# USB 检测：按平台查找 config.txt
# ===========================================
def _config_has_var(config_path):
    if not config_path or not os.path.exists(config_path):
        return False
    try:
        try:
            with io.open(config_path, "r", encoding="utf-8", errors="ignore") as f:
                content = f.read()
        except Exception:
            with io.open(config_path, "r") as f:
                content = f.read()
        pattern = r"^" + re.escape(UIConfig.VAR_NAME) + r"\s+\S+"
        return re.search(pattern, content, re.MULTILINE) is not None
    except Exception:
        return False


def find_usb_with_ctypes():
    try:
        import ctypes
    except Exception:
        return None
    candidates = []
    try:
        for letter in "ABCDEFGHIJKLMNOPQRSTUVWXYZ":
            root = letter + ":\\"
            if not os.path.exists(root):
                continue
            try:
                if ctypes.windll.kernel32.GetDriveTypeW(root) != 2:
                    continue
            except Exception:
                continue
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(cfg)
    except Exception:
        return None
    return candidates or None


def find_usb_with_java():
    try:
        import java.io as jio
    except:
        return None
    candidates = []
    try:
        roots = jio.File.listRoots()
        for r in roots:
            root = r.getAbsolutePath()
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(cfg)
    except:
        return None
    return candidates or None


def find_usb_on_macos():
    try:
        vols_root = '/Volumes'
        if not os.path.exists(vols_root):
            return None
    except Exception:
        return None
    candidates = []
    try:
        for name in os.listdir(vols_root):
            path = os.path.join(vols_root, name)
            cfg = os.path.join(path, 'config.txt')
            if os.path.exists(cfg) and _config_has_var(cfg):
                root = path if path.endswith(os.sep) else path + os.sep
                candidates.append(os.path.join(root, 'config.txt'))
    except Exception:
        return None
    return candidates or None


def find_usb():
    global _USB_FIND_STATUS
    _USB_FIND_STATUS = None
    candidates = []
    try:
        r = find_usb_with_ctypes()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass

    try:
        from java.lang import System
        osname = System.getProperty('os.name')
    except Exception:
        osname = None

    if osname and 'mac' in osname.lower():
        try:
            r = find_usb_on_macos()
            if r:
                candidates.extend(list(r))
        except Exception:
            pass

    try:
        r = find_usb_with_java()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass

    unique = []
    seen = set()
    for p in candidates:
        try:
            key = p.lower()
        except Exception:
            key = p
        if key not in seen:
            seen.add(key)
            unique.append(p)

    if not unique:
        _USB_FIND_STATUS = "none"
        return None
    if len(unique) > 1:
        _USB_FIND_STATUS = "multiple"
        try:
            if _LANG == "en":
                msg = u"Multiple USB drives with config.txt found. Please check."
                title = u"Multiple USB Found"
            else:
                msg = u"找到多个含有config.txt的U盘，请检查。"
                title = u"找到多个U盘"
            JOptionPane.showMessageDialog(None, msg, title, JOptionPane.WARNING_MESSAGE)
        except Exception:
            pass
        return None
    _USB_FIND_STATUS = "ok"
    return unique[0]


def _get_machine_controls():
    try:
        frame = MainFrame.get()
        if frame is None:
            return None
        return frame.getMachineControls()
    except Exception:
        return None


def _run_xy_park_action():
    if Configuration is None or UiUtils is None or MovableUtils is None:
        return False
    panel = _get_machine_controls()
    if panel is None:
        return False
    try:
        def _task():
            tool = panel.getSelectedTool()
            head = None
            try:
                head = tool.getHead() if tool is not None else None
            except Exception:
                head = None
            if head is None:
                head = Configuration.get().getMachine().getDefaultHead()
            MovableUtils.park(head)
            try:
                MovableUtils.fireTargetedUserAction(head.getDefaultHeadMountable())
            except Exception:
                pass
        UiUtils.submitUiMachineTask(_task)
        return True
    except Exception:
        return False


def _get_selected_tool_name(panel):
    try:
        tool = panel.getSelectedTool()
    except Exception:
        tool = None
    if tool is None:
        return None, None
    try:
        name = tool.getName()
    except Exception:
        try:
            name = str(tool)
        except Exception:
            name = None
    return tool, name


def _get_z_value(panel):
    try:
        loc = panel.getCurrentLocation()
    except Exception:
        loc = None
    if loc is None:
        return None
    try:
        return float(loc.getZ())
    except Exception:
        return None


# ===========================================
# 主应用：UI 结构与业务逻辑
# ===========================================
class CalculatorApp(object):
    def __init__(self,
                 input_field_width=UIConfig.INPUT_FIELD_WIDTH,
                 label_width=UIConfig.LABEL_WIDTH,
                 button_width=UIConfig.BUTTON_WIDTH,
                 h_gap=UIConfig.H_GAP,
                 content_width=UIConfig.CONTENT_WIDTH,
                 calc_btn_color=UIConfig.CALC_BUTTON_COLOR,
                 clear_btn_color=UIConfig.CLEAR_BUTTON_COLOR,
                 rewrite_btn_color=UIConfig.REWRITE_BUTTON_COLOR,
                 title_font_size=UIConfig.TITLE_FONT_SIZE,
                 title_color=UIConfig.TITLE_COLOR):
        # 参数说明 / Parameters:
        # - input_field_width: 输入框首选宽度（px）；窗口变宽时允许扩展 / preferred width (px); can expand on resize
        # - label_width: 每行左侧标签固定宽度（px） / fixed label width (px)
        # - button_width: 右侧 Read 按钮固定宽度（px） / fixed width for right 'Read' buttons (px)
        # - h_gap: 组件之间固定水平间距（px） / fixed horizontal gap between components (px)
        # - content_width: 内容面板统一宽度（px） / canonical width for content panels (px)
        #   面板用此保持对齐，输入行受此约束 / Panels align to this; input rows constrained to it
        #   从而与结果面板左右对齐 / so they align left/right with result panel
        # - *_btn_color: 可选按钮背景色 / optional button background color (java.awt.Color or (r,g,b))
        # - title_font_size: 标题字体大小 / title font size
        # - title_color: 标题颜色（默认黑色） / title color (default black)
        #
        # 设计说明 / Design notes:
        # - 使用首选尺寸初始化布局，允许输入框扩展 / use preferred sizes and allow fields to expand
        #   设置很大的最大宽度使其自动扩展 / large max width makes fields passively expand
        #   当窗口更宽时填充水平空间 / fill horizontal space when panel is wider
        #   超过首选宽度 / beyond preferred sizes
        # - 输入区与结果区共享 content_width / input and result panels share content_width
        #   以保持左右对齐 / to align left and right

        self.input_field_width = input_field_width
        self.label_width = label_width
        self.button_width = button_width
        self.h_gap = h_gap
        # 内容宽度（用于 result/history/input_panel 统一宽度） / content width for result/history/input_panel
        self.content_width = content_width
        # 按钮颜色参数（可为 (r,g,b) 或 java.awt.Color） / button color (tuple or java.awt.Color)
        self.calc_btn_color = calc_btn_color
        self.clear_btn_color = clear_btn_color
        self.rewrite_btn_color = rewrite_btn_color
        # 标题样式 / title style
        self.title_font_size = title_font_size
        self.title_color = title_color if title_color is not None else Color(0, 0, 0)
        self.ui_font_name = _pick_font_family(UIConfig.FONT_FALLBACKS) or "Dialog"
        self.lang = UIConfig.DEFAULT_LANG
        self.texts = _load_translations(self.lang)
        self.texts_zh = _load_translations("zh")
        self.texts_en = _load_translations("en")

        self.history_model = DefaultTableModel(
            ["No.", "Current Balance", "N1 Height", "N2 Height", "New Balance", "Status"], 0
        )
        self.calc_count = 0
        self.frame = None
        self.usb_cfg = None
        self.last_new_balance = None
        self._result_locked = False
        self.breath_timer = None
        self.breath_start_ms = 0
        self.rewrite_base_color = None
        self._build_ui()
        self.apply_language()

    # ---------------------- 数学辅助 / Math helpers ----------------------
    def float_equal(self, a, b):
        return abs(a - b) < UIConfig.FLOAT_TOLERANCE

    def float_lte(self, value, threshold):
        return self.float_equal(value, threshold) or value < threshold

    # ---------------------- USB 辅助 / USB helpers ------------------------
    def _detect_usb_legacy(self):
        cfg = find_usb()
        if cfg:
            self.usb_cfg = cfg
            print("Found usb config:", cfg)
        else:
            self.usb_cfg = None
            print("No matching USB config discovered at startup.")
            # 自动识别失败时不再弹出确认对话框 / No confirm dialog on auto-detect failure

    def _detect_usb(self):
        global _USB_SELECTED_ROOT
        try:
            if _USB_SELECTED_ROOT and os.path.exists(_USB_SELECTED_ROOT):
                self.usb_cfg = _USB_SELECTED_ROOT
                return True
        except Exception:
            pass
        cfg = find_usb()
        if cfg:
            self.usb_cfg = cfg
            try:
                _USB_SELECTED_ROOT = cfg
            except Exception:
                pass
            return True
        self.usb_cfg = None
        return False

    def _schedule_auto_usb_find(self):
        try:
            delay_s = float(getattr(UIConfig, "AUTO_USB_FIND_DELAY_SECONDS", 0))
        except Exception:
            delay_s = 0
        if delay_s <= 0:
            return
        try:
            delay_ms = int(delay_s * 1000)
        except Exception:
            delay_ms = 0
        if delay_ms <= 0:
            return
        try:
            print("Auto USB find scheduled: %ss" % delay_s)
        except Exception:
            pass

        def _run(event=None):
            try:
                try:
                    print("Auto USB find triggered")
                except Exception:
                    pass
                if not self.usb_cfg:
                    try:
                        print("Auto USB find: calling _detect_usb()")
                    except Exception:
                        pass
                    self._detect_usb()
                    if _USB_FIND_STATUS == "none":
                        try:
                            msg = self.t("msg.no_usb")
                            title = self.t("dialog.usb")
                            JOptionPane.showMessageDialog(self.frame, msg, title, JOptionPane.WARNING_MESSAGE)
                        except Exception:
                            pass
                    if _USB_FIND_STATUS == "multiple":
                        try:
                            if self.lang == "en":
                                msg = u"Multiple USB drives with config.txt found. Please check."
                                title = u"Multiple USB Found"
                            else:
                                msg = u"找到多个含有config.txt的U盘，请检查。"
                                title = u"找到多个U盘"
                            JOptionPane.showMessageDialog(self.frame, msg, title, JOptionPane.WARNING_MESSAGE)
                        except Exception:
                            pass
                else:
                    try:
                        print("Auto USB find: usb_cfg already set: %s" % str(self.usb_cfg))
                    except Exception:
                        pass
            except Exception:
                pass

        try:
            t = Timer(delay_ms, _run)
            try:
                t.setRepeats(False)
            except Exception:
                pass
            t.start()
            try:
                print("Auto USB find timer started (%sms)" % delay_ms)
            except Exception:
                pass
        except Exception:
            pass

    def _choose_config_file(self):
        try:
            chooser = JFileChooser()
            chooser.setDialogTitle(self.t("dialog.select_config.title"))
            chooser.setFileSelectionMode(JFileChooser.FILES_ONLY)
            # 允许用户选择任意文件（不强制过滤） / Allow any file selection (no strict filter)
            if chooser.showOpenDialog(self.frame) == JFileChooser.APPROVE_OPTION:
                selected = chooser.getSelectedFile()
                if selected and os.path.exists(selected.getAbsolutePath()):
                    # 将用户选择路径作为 usb_cfg / Use selected path as usb_cfg
                    self.usb_cfg = selected.getAbsolutePath()
                    print('Manually selected config:', self.usb_cfg)
                    return True
        except Exception:
            pass
        return False

    def _read_variable_from_config(self, varname):
        if not self.usb_cfg or not os.path.exists(self.usb_cfg):
            return None
        try:
            with io.open(self.usb_cfg, 'r', encoding='utf-8') as f:
                content = f.read()
        except Exception:
            return None
        m = re.search(r'^' + re.escape(varname) + r'\s+(\S+)', content, re.MULTILINE)
        if m:
            return m.group(1)
        return None

    def _replace_variable_in_config(self, varname, newvalue):
        # 仅替换首个匹配并保留尾部注释 / Replace only the first match, keep following comments
        if not self.usb_cfg or not os.path.exists(self.usb_cfg):
            return False, self.t("msg.usb_config_not_found")
        try:
            with io.open(self.usb_cfg, 'r', encoding='utf-8') as f:
                content = f.read()
            pattern = r'^((' + re.escape(varname) + r'))(\s+)(\S+)(.*)$'
            def _repl_var(m):
                return m.group(1) + m.group(3) + str(newvalue) + m.group(5)
            new = re.sub(pattern, _repl_var, content, count=1, flags=re.MULTILINE)
            with io.open(self.usb_cfg, 'w', encoding='utf-8') as f:
                f.write(new)
            return True, None
        except Exception as e:
            return False, str(e)

    # ---------------------- UI 辅助 / UI helpers -----------------------
    def _build_ui(self):
        frame = JFrame("Z-Axis Balancing Calculator")
        frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE)
        # 初始首选尺寸；面板随窗口伸缩 / Initial preferred size; panels expand/shrink with frame
        frame.setPreferredSize(Dimension(self.content_width + UIConfig.FRAME_WIDTH_PAD, UIConfig.FRAME_HEIGHT))

        # 根容器使用 BorderLayout 填充宽度 / Root uses BorderLayout to fill width
        # 避免直接使用 BoxLayout 根容器造成左侧空隙 / Avoid left gap from BoxLayout root
        root = JPanel()
        root.setLayout(BorderLayout())
        root.setBorder(BorderFactory.createEmptyBorder(12, 12, 12, 12))

        # 内容区垂直堆叠并放在 CENTER / Content stacks vertically in CENTER
        content = JPanel()
        content.setLayout(BoxLayout(content, BoxLayout.Y_AXIS))

        title_row = JPanel()
        title_row.setLayout(BorderLayout())
        title_row.setAlignmentX(JPanel.LEFT_ALIGNMENT)
        title_row.setPreferredSize(Dimension(self.content_width, UIConfig.TITLE_HEIGHT))
        try:
            title_row.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.TITLE_HEIGHT))
        except Exception:
            title_row.setMaximumSize(Dimension(self.content_width, UIConfig.TITLE_HEIGHT))

        self.title_label = JLabel("Z-Axis Balancing Calculator")
        self.title_label.setFont(Font(self.ui_font_name, Font.BOLD, self.title_font_size))
        self.title_label.setForeground(self.title_color)
        self.title_label.setHorizontalAlignment(JLabel.CENTER)
        title_row.add(self.title_label, BorderLayout.CENTER)

        lang_panel = JPanel()
        lang_panel.setLayout(BoxLayout(lang_panel, BoxLayout.X_AXIS))
        self.btn_lang_zh = JButton(self.t(UIConfig.LANG_LABEL_ZH, "CN"), actionPerformed=self.on_lang_zh)
        self.btn_lang_en = JButton(self.t(UIConfig.LANG_LABEL_EN, "EN"), actionPerformed=self.on_lang_en)
        self.btn_lang_zh.setPreferredSize(Dimension(UIConfig.LANG_BUTTON_WIDTH, UIConfig.ROW_HEIGHT))
        self.btn_lang_en.setPreferredSize(Dimension(UIConfig.LANG_BUTTON_WIDTH, UIConfig.ROW_HEIGHT))
        lang_panel.add(self.btn_lang_zh)
        lang_panel.add(Box.createRigidArea(Dimension(UIConfig.H_GAP, 0)))
        lang_panel.add(self.btn_lang_en)
        title_row.add(lang_panel, BorderLayout.EAST)

        content.add(title_row)
        content.add(Box.createRigidArea(Dimension(0, UIConfig.TITLE_GAP)))

        # 依次添加面板并统一首选宽度 / Add panels and unify preferred widths
        # 使用 content_width 并允许扩展 / use content_width and allow expansion
        # 随窗口伸缩 / when frame resizes
        content.add(self._input_panel())
        content.add(self._button_panel())
        content.add(self._result_panel())
        content.add(self._history_panel())

        root.add(content, BorderLayout.CENTER)

        frame.setContentPane(root)
        frame.pack()
        frame.setLocationRelativeTo(None)
        self.frame = frame

    def t(self, key, default=None):
        try:
            return self._to_unicode(self.texts.get(key, default if default is not None else key))
        except Exception:
            return self._to_unicode(default if default is not None else key)

    def _set_rewrite_enabled(self, enabled):
        try:
            self.rewrite_btn.setEnabled(enabled)
        except Exception:
            return
        if enabled:
            self._start_breath()
        else:
            self._stop_breath()

    def _set_action_buttons_enabled(self, enabled):
        try:
            self.read_balance_btn.setEnabled(enabled)
        except Exception:
            pass
        try:
            self.read_n1_btn.setEnabled(enabled)
        except Exception:
            pass
        try:
            self.read_n2_btn.setEnabled(enabled)
        except Exception:
            pass
        try:
            self.calc_btn.setEnabled(enabled)
        except Exception:
            pass

    def _start_breath(self):
        if self.rewrite_btn is None:
            return
        if self.rewrite_base_color is None:
            try:
                self.rewrite_base_color = self.rewrite_btn.getBackground()
            except Exception:
                self.rewrite_base_color = None
        if self.breath_timer is None:
            self.breath_start_ms = java.lang.System.currentTimeMillis()

            def _tick(event=None):
                try:
                    period = float(UIConfig.REWRITE_BREATH_PERIOD_MS)
                    if period <= 0:
                        return
                    now_ms = java.lang.System.currentTimeMillis()
                    phase = (now_ms - self.breath_start_ms) / period
                    intensity = (math.sin(phase * 2.0 * math.pi) + 1.0) / 2.0
                    base = _to_color(self.rewrite_base_color, Color(200, 200, 200))
                    target = _to_color(UIConfig.REWRITE_BREATH_COLOR, Color(0, 176, 80))
                    r = int(base.getRed() + (target.getRed() - base.getRed()) * intensity)
                    g = int(base.getGreen() + (target.getGreen() - base.getGreen()) * intensity)
                    b = int(base.getBlue() + (target.getBlue() - base.getBlue()) * intensity)
                    self.rewrite_btn.setBackground(Color(r, g, b))
                except Exception:
                    pass

            self.breath_timer = Timer(UIConfig.REWRITE_BREATH_INTERVAL_MS, _tick)
            try:
                self.breath_timer.start()
            except Exception:
                self.breath_timer = None
        else:
            try:
                if not self.breath_timer.isRunning():
                    self.breath_timer.start()
            except Exception:
                pass

    def _stop_breath(self):
        if self.breath_timer is not None:
            try:
                self.breath_timer.stop()
            except Exception:
                pass
        self.breath_timer = None
        if self.rewrite_base_color is not None:
            try:
                self.rewrite_btn.setBackground(self.rewrite_base_color)
            except Exception:
                pass

    def t_lang(self, lang, key, default=None):
        try:
            if lang == "zh":
                texts = self.texts_zh
            else:
                texts = self.texts_en
        except Exception:
            texts = None
        try:
            if texts:
                return self._to_unicode(texts.get(key, default if default is not None else key))
        except Exception:
            pass
        return self._to_unicode(default if default is not None else key)

    def _to_unicode(self, value):
        try:
            if isinstance(value, unicode):
                return value
        except Exception:
            pass
        try:
            return self._decode_escapes(unicode(value))
        except Exception:
            pass
        try:
            return self._decode_escapes(str(value).decode("utf-8"))
        except Exception:
            return u""

    def _decode_escapes(self, text):
        try:
            if text and u"\\u" in text:
                try:
                    return text.encode("latin-1").decode("unicode_escape")
                except Exception:
                    try:
                        return text.decode("unicode_escape")
                    except Exception:
                        return text
        except Exception:
            pass
        return text

    def set_language(self, lang):
        self.lang = lang
        self.texts = _load_translations(lang)
        self.apply_language()

    def apply_language(self):
        # 标题 / Title
        try:
            self.title_label.setText(self.t("title.label"))
            self.frame.setTitle(self.t("title.frame"))
        except Exception:
            pass
        try:
            self.btn_lang_zh.setText(self.t(UIConfig.LANG_LABEL_ZH, "CN"))
            self.btn_lang_en.setText(self.t(UIConfig.LANG_LABEL_EN, "EN"))
        except Exception:
            pass
        # 面板标题 / Panel titles
        try:
            self.input_panel_border.setTitle(self.t("panel.inputs"))
            self.result_panel_border.setTitle(self.t("panel.result"))
            self.history_panel_border.setTitle(self.t("panel.history"))
        except Exception:
            pass
        # 表头 / Table headers
        try:
            self.history_model.setColumnIdentifiers([
                self.t("col.no"),
                self.t("col.balance"),
                self.t("col.n1"),
                self.t("col.n2"),
                self.t("col.new_balance"),
                self.t("col.status"),
            ])
        except Exception:
            pass
        # ????????? / Update status column when language changes
        try:
            status_col = 5
            zh_pending = self.texts_zh.get("result.pending", u"????")
            en_pending = self.texts_en.get("result.pending", u"Adjust Needed")
            zh_bal = self.texts_zh.get("result.balanced", u"???")
            en_bal = self.texts_en.get("result.balanced", u"Balanced")
            new_pending = self.t("result.pending")
            new_bal = self.t("result.balanced")
            for r in range(self.history_model.getRowCount()):
                val = self.history_model.getValueAt(r, status_col)
                if val == zh_pending or val == en_pending:
                    self.history_model.setValueAt(new_pending, r, status_col)
                elif val == zh_bal or val == en_bal:
                    self.history_model.setValueAt(new_bal, r, status_col)
        except Exception:
            pass
        # 标签与按钮 / Labels and buttons
        try:
            self.balance_label.setText(self.t("label.balance"))
            self.n1_label.setText(self.t("label.n1"))
            self.n2_label.setText(self.t("label.n2"))
            self.read_balance_btn.setText(self.t("button.read_balance"))
            self.read_n1_btn.setText(self.t("button.read_n1"))
            self.read_n2_btn.setText(self.t("button.read_n2"))
            self.calc_btn.setText(self.t("button.calculate"))
            self.clear_btn.setText(self.t("button.clear"))
            self.rewrite_btn.setText(self.t("button.rewrite"))
        except Exception:
            pass
        # 默认结果文本 / Default result text
        try:
            if not self._result_locked:
                self._set_default_result_text()
        except Exception:
            pass
        try:
            self.frame.repaint()
        except Exception:
            pass

    def _set_default_result_text(self):
        guide_title = self.t("guide.title")
        steps = [
            self.t("guide.step1"),
            self.t("guide.step2"),
            self.t("guide.step3"),
            self.t("guide.step4"),
            
        ]
        html_steps = []
        for step in steps:
            if step:
                html_steps.append(u"<span style='font-size:14pt;'>{}</span><br>".format(step))
        html = u"<html><span style='font-size:22pt;'><b style='color:green'>{}</b></span><br><br>{}</html>".format(
            guide_title, "".join(html_steps)
        )
        self.result_area.setText(html)

    def on_lang_zh(self, event=None):
        self.set_language("zh")

    def on_lang_en(self, event=None):
        self.set_language("en")

    def _input_panel(self):
        panel = JPanel()
        panel.setLayout(BoxLayout(panel, BoxLayout.Y_AXIS))
        panel.setBorder(BorderFactory.createTitledBorder("Inputs - Negative values required"))
        self.input_panel_border = panel.getBorder()
        # 左对齐输入面板并统一宽度 / Left-align input panel and unify width
        panel.setAlignmentX(JPanel.LEFT_ALIGNMENT)
        # 首选宽度与结果区一致 / Preferred width matches result area
        # 最大宽度允许扩展 / Max width allows expansion
        # 左右边缘随窗口变化 / Left/right edges track window size
        panel.setPreferredSize(Dimension(self.content_width, UIConfig.INPUT_PANEL_HEIGHT))
        try:
            panel.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.INPUT_PANEL_HEIGHT))
        except Exception:
            panel.setMaximumSize(Dimension(32767, UIConfig.INPUT_PANEL_HEIGHT))

        # 手动构建三行以加入 Read 按钮 / Build three rows to add Read buttons
        # 输入框宽度使用配置值 / Input field width comes from config
        def make_row(label_text, read_label, read_action):
            row = JPanel()
            row.setLayout(BoxLayout(row, BoxLayout.X_AXIS))
            # 行内左对齐 / Left-align row
            row.setAlignmentX(JPanel.LEFT_ALIGNMENT)
            # 首选宽度与内容区一致 / Preferred width aligns to content area
            # 最大宽度允许行扩展 / Max width allows row expansion
            row.setPreferredSize(Dimension(self.content_width, UIConfig.ROW_HEIGHT))
            try:
                row.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.ROW_HEIGHT))
            except Exception:
                row.setMaximumSize(Dimension(32767, UIConfig.ROW_HEIGHT))

            lbl = JLabel(label_text)
            # 标签固定宽度以对齐输入框和按钮 / Fixed label width aligns field and button
            lbl.setPreferredSize(Dimension(self.label_width, UIConfig.ROW_HEIGHT))
            try:
                # 保持标签最大宽度固定 / Keep label maximum fixed
                lbl.setMaximumSize(Dimension(self.label_width, UIConfig.ROW_HEIGHT))
            except Exception:
                lbl.setMaximumSize(Dimension(self.label_width, UIConfig.ROW_HEIGHT))
            lbl.setAlignmentX(JLabel.LEFT_ALIGNMENT)
            try:
                lbl.setFont(Font(self.ui_font_name, Font.PLAIN, UIConfig.LABEL_FONT_SIZE))
            except Exception:
                pass

            field = JTextField()
            # 首选宽度为默认值 / Preferred width is nominal
            # 最大宽度允许扩展 / Max width allows expansion
            field.setPreferredSize(Dimension(self.input_field_width, UIConfig.ROW_HEIGHT))
            try:
                field.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.ROW_HEIGHT))
            except Exception:
                field.setMaximumSize(Dimension(32767, UIConfig.ROW_HEIGHT))
            # 设置最小宽度避免消失 / Set minimum width to avoid collapse
            field.setMinimumSize(Dimension(50, UIConfig.ROW_HEIGHT))
            field.setAlignmentX(JTextField.LEFT_ALIGNMENT)
            try:
                field.setFont(Font(self.ui_font_name, Font.PLAIN, UIConfig.INPUT_FONT_SIZE))
            except Exception:
                pass

            read_btn = JButton(read_label, actionPerformed=read_action)
            # 输入框与按钮之间添加间距 / Add spacing between field and button
            row.add(lbl)
            # 输入框占据剩余空间 / Field takes remaining space
            row.add(field)
            # 添加 glue 将按钮推到右侧 / Add glue to push button right
            # glue + 大最大宽度使输入框自动扩展 / glue + large max width expands field
            # 填充标签与按钮之间区域 / fill space between label and button
            row.add(Box.createHorizontalGlue())
            # 按钮前留固定间距 / Add fixed gap before button
            row.add(Box.createRigidArea(Dimension(self.h_gap, 0)))

            # 统一按钮尺寸（glue 会将按钮推到右侧） / Uniform button size (glue pushes right)
            read_btn.setPreferredSize(Dimension(self.button_width, UIConfig.ROW_HEIGHT))
            read_btn.setMaximumSize(Dimension(self.button_width, UIConfig.ROW_HEIGHT))
            read_btn.setMinimumSize(Dimension(self.button_width, UIConfig.ROW_HEIGHT))
            read_btn.setAlignmentX(JButton.LEFT_ALIGNMENT)

            # 设置按钮颜色（若提供 RGB 元组） / Set button color (if RGB tuple provided)
            try:
                if self.calc_btn_color and read_label == 'Read Balance':
                    if isinstance(self.calc_btn_color, tuple):
                        read_btn.setBackground(Color(self.calc_btn_color[0], self.calc_btn_color[1], self.calc_btn_color[2]))
                    else:
                        read_btn.setBackground(self.calc_btn_color)
            except Exception:
                pass
            row.add(read_btn)
            panel.add(row)
            return field, read_btn, lbl

        # 使用配置参数创建三行输入 / Create three input rows from config
        self.balance_field, self.read_balance_btn, self.balance_label = make_row("Current Balance: ", "Read Balance", self.on_read_balance)
        panel.add(Box.createRigidArea(Dimension(0, UIConfig.INPUT_ROW_GAP)))
        self.n1_field, self.read_n1_btn, self.n1_label = make_row("N1 Height (<= 0):", "Read N1", self.on_read_n1)
        panel.add(Box.createRigidArea(Dimension(0, UIConfig.INPUT_ROW_GAP)))
        self.n2_field, self.read_n2_btn, self.n2_label = make_row("N2 Height (<= 0):", "Read N2", self.on_read_n2)

        return panel

    # _labeled_field 已保留但未使用 / _labeled_field kept but unused

    def _button_panel(self):
        # 水平居中按钮区域 / Centered horizontal button panel
        panel = JPanel()
        panel.setLayout(BoxLayout(panel, BoxLayout.X_AXIS))
        panel.setBorder(BorderFactory.createEmptyBorder(8, 0, 8, 0))
        # 按钮面板与其他面板对齐并允许扩展 / Align button panel and allow expansion
        panel.setAlignmentX(JPanel.LEFT_ALIGNMENT)
        panel.setPreferredSize(Dimension(self.content_width, UIConfig.BUTTON_PANEL_HEIGHT))
        try:
            panel.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.BUTTON_PANEL_HEIGHT))
        except Exception:
            panel.setMaximumSize(Dimension(self.content_width, UIConfig.BUTTON_PANEL_HEIGHT))
        panel.add(Box.createHorizontalGlue())

        btn_container = JPanel()
        btn_container.setLayout(BoxLayout(btn_container, BoxLayout.X_AXIS))

        self.calc_btn = JButton("Calculate", actionPerformed=self.on_calculate)
        self.clear_btn = JButton("Clear", actionPerformed=self.on_clear)
        rewrite_btn = JButton("Rewrite Balance", actionPerformed=self.on_rewrite)
        rewrite_btn.setEnabled(False)
        self.rewrite_btn = rewrite_btn
        try:
            self.rewrite_base_color = rewrite_btn.getBackground()
        except Exception:
            self.rewrite_base_color = None

        # 统一设置按钮尺寸并应用颜色（若有） / Uniform button size and apply color (if any)
        for b, color in ((self.calc_btn, self.calc_btn_color), (self.clear_btn, self.clear_btn_color), (rewrite_btn, self.rewrite_btn_color)):
            b.setPreferredSize(Dimension(self.button_width, UIConfig.BUTTON_HEIGHT))
            b.setMaximumSize(Dimension(self.button_width, UIConfig.BUTTON_HEIGHT))
            b.setMinimumSize(Dimension(self.button_width, UIConfig.BUTTON_HEIGHT))
            try:
                if color is not None:
                    if isinstance(color, tuple):
                        b.setBackground(Color(color[0], color[1], color[2]))
                    else:
                        b.setBackground(color)
            except Exception:
                pass
            btn_container.add(b)
            btn_container.add(Box.createRigidArea(Dimension(UIConfig.BUTTON_GAP, 0)))

        panel.add(btn_container)
        panel.add(Box.createHorizontalGlue())
        return panel

    def _result_panel(self):
        panel = JPanel()
        panel.setLayout(BoxLayout(panel, BoxLayout.Y_AXIS))
        panel.setBorder(BorderFactory.createTitledBorder("Result"))
        self.result_panel_border = panel.getBorder()
        # 与其他面板左边缘对齐 / Align left edge with other panels
        panel.setAlignmentX(JPanel.LEFT_ALIGNMENT)
        panel.setPreferredSize(Dimension(self.content_width, UIConfig.RESULT_PANEL_HEIGHT))
        try:
            panel.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.RESULT_PANEL_HEIGHT))
        except Exception:
            panel.setMaximumSize(Dimension(self.content_width, UIConfig.RESULT_PANEL_HEIGHT))

        self.result_area = JTextPane()
        self.result_area.setEditable(False)
        self.result_area.setContentType("text/html")
        # 首选宽度用于初始布局 / Preferred width for initial layout
        # 最大宽度允许扩展以保持对齐 / Max width allows expansion to keep alignment
        self.result_area.setPreferredSize(Dimension(self.content_width, UIConfig.RESULT_AREA_HEIGHT))
        try:
            self.result_area.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.RESULT_AREA_HEIGHT))
        except Exception:
            self.result_area.setMaximumSize(Dimension(32767, 260))
        # self.result_area.setText("<html><i>No result yet.</i></html>")  # 旧版占位文本 / legacy placeholder
        self._set_default_result_text()

        panel.add(self.result_area)
        return panel

    def _history_panel(self):
        panel = JPanel()
        panel.setLayout(BoxLayout(panel, BoxLayout.Y_AXIS))
        panel.setBorder(BorderFactory.createTitledBorder("Last 5 Records"))
        self.history_panel_border = panel.getBorder()
        panel.setAlignmentX(JPanel.LEFT_ALIGNMENT)
        panel.setPreferredSize(Dimension(self.content_width, UIConfig.HISTORY_PANEL_HEIGHT))
        try:
            panel.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.HISTORY_PANEL_HEIGHT))
        except Exception:
            panel.setMaximumSize(Dimension(self.content_width, UIConfig.HISTORY_PANEL_HEIGHT))

        table = JTable(self.history_model)
        # 创建单元格渲染器并居中对齐 / Create a renderer and center-align
        renderer = DefaultTableCellRenderer()
        renderer.setHorizontalAlignment(JLabel.CENTER)
        # 设置所有单元格默认居中 / Set default renderer to center
        try:
            table.setDefaultRenderer(java.lang.Object, renderer)
        except Exception:
            pass
        # 为每一列应用居中渲染器 / Apply center renderer to each column
        for i in range(self.history_model.getColumnCount()):
            table.getColumnModel().getColumn(i).setCellRenderer(renderer)
        # 表头居中 / Center table header
        table.getTableHeader().setDefaultRenderer(renderer)
        # 首列为其他列的一半，总宽度为 content_width / First column half width; total is content_width
        try:
            col_count = self.history_model.getColumnCount()
            if col_count > 1:
                total = self.content_width
                unit = int(total / UIConfig.HISTORY_COL_UNIT_DIVISOR)  # 0.5 + 5*1 = 5.5 units
                # No. 列宽 / No. column width
                table.getColumnModel().getColumn(0).setPreferredWidth(int(unit * 0.5))
                # 其余列宽度（1..5） / Remaining column widths (1..5)
                for idx in range(1, col_count):
                    table.getColumnModel().getColumn(idx).setPreferredWidth(unit)
        except Exception:
            pass

        scroll = JScrollPane(table)
        # 历史面板宽度与内容区一致并允许扩展 / History panel aligns and can expand
        scroll.setPreferredSize(Dimension(self.content_width, UIConfig.HISTORY_SCROLL_HEIGHT))
        try:
            scroll.setMaximumSize(Dimension(java.lang.Integer.MAX_VALUE, UIConfig.HISTORY_SCROLL_HEIGHT))
        except Exception:
            scroll.setMaximumSize(Dimension(32767, 200))

        panel.add(scroll)
        return panel

    # ---------------------- 操作 / Actions --------------------------
    def on_clear(self, event=None):
        for field in (self.balance_field, self.n1_field, self.n2_field):
            field.setText("")
            field.setBackground(Color.white)
        # self.result_area.setText("<html><i>No result yet.</i></html>")  # 旧版占位文本 / legacy placeholder
        self._result_locked = False
        self._set_default_result_text()
        self.balance_field.requestFocusInWindow()
        # 禁用重写并清空缓存值 / Disable rewrite and clear cached value
        self._set_rewrite_enabled(False)
        self._set_action_buttons_enabled(True)
        self.last_new_balance = None

    def on_calculate(self, event=None):
        try:
            balance = self._parse(self.balance_field, "label.balance", "Current Balance")
            n1 = self._parse(self.n1_field, "label.n1", "N1 Height")
            n2 = self._parse(self.n2_field, "label.n2", "N2 Height")
        except ValueError as ex:
            JOptionPane.showMessageDialog(self.frame, self._to_unicode(ex), self.t("dialog.input_error"), JOptionPane.ERROR_MESSAGE)
            return

        diff = n1 - n2
        abs_diff = abs(diff)
        texts = {
            "balanced": self.t("result.balanced"),
            "pending": self.t("result.pending"),
            "conclusion": self.t("result.conclusion"),
            "new_balance": self.t("result.new_balance"),
            "guide_title": self.t("guide.title"),
            "guide_steps": [
                self.t("guide.step6"),
            ]
        }

        is_adjust_needed = not self.float_lte(abs_diff, UIConfig.THRESHOLD)
        if not is_adjust_needed:
            message = u"<h2>{}</h2>".format(texts["conclusion"])
            status = texts["balanced"]
            new_balance_val = None
            self._result_locked = False
        else:
            adjustment = abs_diff * UIConfig.ADJUST_FACTOR
            new_balance_val = balance + adjustment if diff < 0 else balance - adjustment
            guide_html = u"<div><h3>{}</h3><ul>{}</ul></div>".format(
                texts["guide_title"],
                u"".join([u"<li>{}</li>".format(step) for step in texts["guide_steps"] if step])
            )
            message = u"<h1>{} <span style='color:red'>{}</span></h1>".format(
                texts["new_balance"], u"{:.3f}".format(new_balance_val)
            ) + guide_html
            status = texts["pending"]
            self._result_locked = True

        self.result_area.setText(u"<html>{}</html>".format(message))
        self._add_history(balance, n1, n2, new_balance_val, status)

        # 仅在需要调整时启用重写 / Enable rewrite only when adjustment needed
        if is_adjust_needed and new_balance_val is not None:
            self._set_rewrite_enabled(True)
            self._set_action_buttons_enabled(False)
            self.last_new_balance = new_balance_val
        else:
            self._set_rewrite_enabled(False)
            self._set_action_buttons_enabled(True)
            self.last_new_balance = None

    def _parse(self, field, label_key, en_label):
        text = field.getText().strip()
        if not text:
            field.setBackground(Color(255, 230, 230))
            label_local = self.t(label_key, en_label)
            msg = self.t("error.required", u"{label} is required.").format(label=label_local)
            raise ValueError(msg)
        try:
            val = float(text)
        except Exception:
            field.setBackground(Color(255, 230, 230))
            label_local = self.t(label_key, en_label)
            msg = self.t("error.number", u"{label} must be a number.").format(label=label_local)
            raise ValueError(msg)
        if val > 0:
            field.setBackground(Color(255, 230, 230))
            label_local = self.t(label_key, en_label)
            msg = self.t("error.nonpositive", u"{label} must be <= 0.").format(label=label_local)
            raise ValueError(msg)
        field.setBackground(Color.white)
        return val

    def _add_history(self, balance, n1, n2, new_balance_val, status):
        self.calc_count += 1
        row = [self.calc_count, self._fmt(balance), self._fmt(n1), self._fmt(n2),
               "-" if new_balance_val is None else self._fmt(new_balance_val),
               status]
        self.history_model.insertRow(0, row)
        if self.history_model.getRowCount() > UIConfig.MAX_HISTORY:
            self.history_model.removeRow(self.history_model.getRowCount() - 1)

    def _fmt(self, value):
        try:
            return "{:.3f}".format(value)
        except Exception:
            return str(value)

    # ---------------- USB 按钮操作 / USB button actions ----------------
    def on_read_balance(self, event=None):
        # 读取 USB config 变量并写入 Current Balance / Read USB variable into Current Balance
        # 参数：event 为 Swing 事件（此处忽略） / Param: event is Swing action event (ignored)
        try:
            if self.usb_cfg and (not os.path.exists(self.usb_cfg)):
                self.usb_cfg = None
            if not self.usb_cfg:
                self.balance_field.setText("")
                self.balance_field.setBackground(Color(255, 230, 230))
                JOptionPane.showMessageDialog(self.frame, self.t("msg.no_usb"), self.t("dialog.usb"), JOptionPane.WARNING_MESSAGE)
                return
            val = self._read_variable_from_config(UIConfig.VAR_NAME)
            if val is None:
                # 未找到变量时清空并提示 / Variable not found -> clear and inform
                self.balance_field.setText("")
                self.balance_field.setBackground(Color(255, 230, 230))
                msg = self.t("msg.var_not_found").format(var=UIConfig.VAR_NAME, path=self.usb_cfg)
                JOptionPane.showMessageDialog(self.frame, msg, self.t("dialog.usb"), JOptionPane.INFORMATION_MESSAGE)
                return
            try:
                # 解析为浮点并格式化 3 位小数 / Parse and format to 3 decimals
                fval = float(val)
                self.balance_field.setText("{:.3f}".format(fval))
                self.balance_field.setBackground(Color.white)
            except Exception:
                # 解析失败时清空并提示 / On parse failure, clear and warn
                self.balance_field.setText("")
                self.balance_field.setBackground(Color(255, 230, 230))
                JOptionPane.showMessageDialog(self.frame, self.t("msg.parse_failed"), self.t("dialog.usb"), JOptionPane.ERROR_MESSAGE)
        except Exception as e:
            try:
                JOptionPane.showMessageDialog(self.frame, unicode(e), self.t("dialog.usb"), JOptionPane.ERROR_MESSAGE)
            except Exception:
                pass

    def on_read_n1(self, event=None):
        # 读取 N1 当前 Z 值 / Read current Z for N1
        panel = _get_machine_controls()
        if panel is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_panel"))
            return
        tool, name = _get_selected_tool_name(panel)
        if name is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_tool"))
            return
        if not name.startswith(UIConfig.N1_PREFIX):
            JOptionPane.showMessageDialog(self.frame, self.t("msg.not_n1"))
            return
        z_val = _get_z_value(panel)
        if z_val is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_panel"))
            return
        if z_val > 0:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.need_neg_n1"))
            return
        self.n1_field.setText("{:.3f}".format(z_val))

    def on_read_n2(self, event=None):
        # 读取 N2 当前 Z 值 / Read current Z for N2
        panel = _get_machine_controls()
        if panel is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_panel"))
            return
        tool, name = _get_selected_tool_name(panel)
        if name is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_tool"))
            return
        if not name.startswith(UIConfig.N2_PREFIX):
            JOptionPane.showMessageDialog(self.frame, self.t("msg.not_n2"))
            return
        z_val = _get_z_value(panel)
        if z_val is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_panel"))
            return
        if z_val > 0:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.need_neg_n2"))
            return
        self.n2_field.setText("{:.3f}".format(z_val))

    def on_rewrite(self, event=None):
        if self.last_new_balance is None:
            JOptionPane.showMessageDialog(self.frame, self.t("msg.no_balance_to_write"), self.t("dialog.rewrite"), JOptionPane.WARNING_MESSAGE)
            return
        if not self.usb_cfg:
            # 自动识别失败时允许手动选择 / Allow manual selection on auto-detect failure
            if not self._choose_config_file():
                JOptionPane.showMessageDialog(self.frame, self.t("msg.no_usb"), self.t("dialog.usb"), JOptionPane.WARNING_MESSAGE)
                return
        # 写入新的 balance 到 config / Write new balance into config
        success, err = self._replace_variable_in_config(UIConfig.VAR_NAME, '{:.3f}'.format(self.last_new_balance))
        if success:
            try:
                _run_xy_park_action()
            except Exception:
                pass
            JOptionPane.showMessageDialog(self.frame, self.t("msg.write_ok"), self.t("dialog.rewrite"), JOptionPane.INFORMATION_MESSAGE)
            # 写入成功后清空界面 / Clear UI after successful write
            self.on_clear()
        else:
            msg = self.t("msg.write_fail").format(err=err)
            JOptionPane.showMessageDialog(self.frame, msg, self.t("dialog.rewrite"), JOptionPane.ERROR_MESSAGE)

    def run(self):
        self.frame.setVisible(True)
        self.balance_field.requestFocusInWindow()
        try:
            SwingUtilities.invokeLater(self._schedule_auto_usb_find)
        except Exception:
            pass


# ===========================================
# 启动入口：构建并显示 UI
# ===========================================
def main():
    app = CalculatorApp()
    app.run()


# OpenPnP 脚本入口：在 EDT 上创建 UI / OpenPnP entry: create UI on EDT
# ===========================================
# OpenPnP 脚本入口：确保在 EDT 中启动
# ===========================================
def start():
    SwingUtilities.invokeLater(main)


# ===========================================
# 直接运行 vs. 脚本菜单启动
# ===========================================
if __name__ == "__main__":
    # 直接执行时也在 EDT 上创建 UI / Ensure UI on EDT when run directly
    SwingUtilities.invokeLater(main)
else:
    # 通过 OpenPnP 脚本菜单启动时立即触发 / When launched via OpenPnP Scripts menu, start() fires immediately
    start()
