#!/usr/bin/env jython

# ===========================================
# 脚本流程概览
# 1) 弹出确认对话框，用户选择是否继续
# 2) 触发 saveConfig 保存配置
# 3) 清理 org.openpnp.* 与历史 log
# 4) 备份 .xml 与 scripts/Events、scripts/TornadoSMT 到 U 盘
# 5) 显示完成提示
# ===========================================

# ===========================================
# 依赖导入：Java / Swing / Python 标准库
# 注意：该脚本以 Jython 运行
# ===========================================
from java.io import File
from java.awt import Desktop, BorderLayout, Font, Dimension
from java.awt.event import ActionListener
from java.net import URI
import traceback
import sys
import os
from javax.swing import SwingUtilities, JDialog, JProgressBar, JTextPane, JScrollPane, JPanel, JButton, BoxLayout, JLabel, Box, JOptionPane
from javax.swing.event import HyperlinkListener
import webbrowser
import subprocess
import shutil
import io, re
import time
import java.lang

# ===========================================
# OpenPnP 可选 API：在运行环境可用时启用
# ===========================================
try:
    from org.openpnp.gui import MainFrame
except Exception:
    MainFrame = None

# ===========================================
# 参数区：集中管理可配置项
# 建议：定制时仅修改此处
# ===========================================
_LANG = "zh"
WEBSITE_URL = "https://www.alandesign.org/"
VAR_NAME = "endstop.maxz.homing_position"  # 要查找的变量名

# ===========================================
# 运行参数
# ===========================================
_PROGRESS_DIALOG = None
_PROGRESS_STATE = {
    "value": None,
    "key": None,
    "default_zh": None,
    "default_en": None,
}
_USB_FIND_STATUS = None
_USB_SELECTED_ROOT = None
_PROGRESS_BAR_HEIGHT = 20
_PROGRESS_BAR_FONT_SIZE = 14
_DIALOG_MSG_FONT_SIZE_PT = 15
_DIALOG_MSG_COLOR = "#D00000"
RESULT_DELAY_SECONDS = 3
LAST_BACKUP_FOLDER_NAME = ""

# ===========================================
# 内置多语言文本
# 作用：用于窗口标题、提示、日志与结果文案
# ===========================================
INLINE_I18N_ZH = {
    "button.continue": u"继续",
    "button.exit": u"退出",
    "dialog.message": u"该维护脚本将对 .openpnp2文件夹 进行一系列必要操作，\n\n过程如下：\n1、清理那些储存着调试图像的文件夹-以 org.openpnp 开头的文件夹；\n2、清理log文件夹中的历史日志文件；\n3、对OpenPnP执行一次保存配置操作；\n4、备份所有.xml文件；\n5、备份scripts目录Events文件夹和TronadoSMT文件夹中的脚本文件；\n6、所有备份文件的存放位置：U盘\TornadoSMT_Backups\年月日_时分秒 文件夹。\n\n确定要继续吗？",
    "dialog.title": u"维护确认",
    "log.title": u"维护日志",
    "final.done": u"已完成：",
    "final.item1": u"清理 org.openpnp.* 文件夹和历史日志；",
    "final.item2": u"备份 .xml文件 + scripts/Events文件夹 + scripts/TornadoSMT文件夹 至贴片机U盘中.",
    "final.next": u"后续操作：",
    "final.step1": u"查看U盘中，最新日期文件夹是否存在以及备份文件是否完整，及时清理陈旧日期的文件夹。",
    "final.step2": u"点击 退出 按钮，关闭此窗口。",
    "final.success": u"维护完成！！",
    "final.tips": u"提示: 备份文件存放在：yyyy-MM-dd_HH.mm.ss文件夹，位于U盘\\TornadoSMT_Backups目录下。",
    "final.more": u"后续更多内容和教程，请访问：",
    "final.link": u" AlanDesign.org ",
    "final.title": u"维护完成",
    "lang.en": u"English",
    "lang.zh": u"中文",
    "none.title": u" ",
    "progress.preparing": u"准备...",
    "progress.saving": u"保存配置中...",
    "progress.cleaning": u"清理中...",
    "progress.backup": u"备份至U盘...",
    "progress.done": u"完成",
    "window.title": u"一键维护 -by AlanDesign",
}

INLINE_I18N_EN = {
    "button.continue": u"Continue",
    "button.exit": u"Exit",
    "dialog.message": u"This script will perform a series of necessary operations on the .openpnp2 folder.\n\nThe contents are as follows:\n1. Clean up folders storing debug images - folders starting with org.openpnp;\n2. Clean up historical log files in the log folder;\n3. Perform Save Configuration operation on OpenPnP;\n4. Back up all .xml files;\n5. Back up script files in the Events folder and TronadoSMT folder under the scripts directory;\n6. Storage location of all backup files: U disk\TornadoSMT_Backups\yyyy-MM-dd_HH.mm.ss folder.\n\nContinue?",
    "dialog.title": u"Maintenance Confirm",
    "log.title": u"Maintenance Log",
    "final.done": u"Done:",
    "final.item1": u"Cleaned org.openpnp.* folders and log files.",
    "final.item2": u"Backed up xml + scripts/Events + scripts/TornadoSMT to U disk.",
    "final.next": u"Next steps:",
    "final.step1": u"Verify backup folder in USB and promptly clean up yyyy-MM-dd_HH.mm.ss folders with outdated dates.",
    "final.step2": u"Close this window by clicking Exit.",
    "final.success": u"Maintenance Completed!!",
    "final.tips": u"Tips: Backup folder name format: yyyy-MM-dd_HH.mm.ss,under U disk\\TornadoSMT_Backups",
    "final.more": u"For more content and tutorials, please visit:",
    "final.link": u" AlanDesign.org ",
    "final.title": u"Maintenance Completed",
    "lang.en": u"English",
    "lang.zh": u"中文",
    "none.title": u" ",
    "progress.preparing": u"Preparing...",
    "progress.saving": u"Saving config...",
    "progress.cleaning": u"Cleaning files...",
    "progress.backup": u"Backing up to U disk...",
    "progress.done": u"Done",
    "window.title": u"One Click Maintenance -by AlanDesign",
}

# ===========================================
# 字体回退：保证中文可读
# ===========================================
_FONT_FALLBACKS = [
    "Microsoft YaHei",
    "PingFang SC",
    "Hiragino Sans GB",
    "Noto Sans CJK SC",
    "WenQuanYi Micro Hei",
    "Dialog",
]


# ===========================================
# 翻译字典：按语言返回文案集合
# ===========================================
def _load_translations(lang):
    if lang == "en":
        return INLINE_I18N_EN
    return INLINE_I18N_ZH


# ===========================================
# 字典取值：key 不存在时回退默认文本
# ===========================================
def _t(props, key, default_text):
    try:
        return props.get(key, default_text)
    except Exception:
        return default_text


# ===========================================
# 字体选择：挑选可显示中文的字体
# ===========================================
def _pick_font_family(fallbacks):
    sample = u"ABC"
    for name in fallbacks:
        try:
            f = Font(name, Font.PLAIN, 12)
            try:
                if f.canDisplayUpTo(sample) == -1:
                    return name
            except Exception:
                try:
                    if f.getFamily() == name:
                        return name
                except Exception:
                    pass
        except Exception:
            pass
    return "Dialog"


# ===========================================
# 字体对象：按字号构造 Font
# ===========================================
def _get_ui_font(size):
    try:
        name = _pick_font_family(_FONT_FALLBACKS)
        return Font(name, Font.PLAIN, int(size))
    except Exception:
        return None


# ===========================================
# HTML 转义：避免文本破坏 HTML 结构
# ===========================================
def _escape_html(text):
    if text is None:
        return ""
    try:
        s = unicode(text)
    except Exception:
        try:
            s = str(text)
        except Exception:
            return ""
    s = s.replace("&", "&amp;")
    s = s.replace("<", "&lt;")
    s = s.replace(">", "&gt;")
    s = s.replace('"', "&quot;")
    s = s.replace("'", "&#39;")
    return s


# ===========================================
# 对话框提示：生成带样式的 HTML
# ===========================================
def _format_dialog_message_html(text):
    font_family = _pick_font_family(_FONT_FALLBACKS)
    safe_text = _escape_html(text)
    safe_text = safe_text.replace("\n", "<br/>")
    html = (
        u"<html><body style='font-family:%s; font-size:%dpt; color:%s;'>%s</body></html>"
        % (font_family, int(_DIALOG_MSG_FONT_SIZE_PT), _DIALOG_MSG_COLOR, safe_text)
    )
    return html


# ===========================================
# 完成页面：拼接最终结果 HTML
# ===========================================
def _get_final_message_html(lang):
    props = _load_translations(lang)
    title = _t(props, "window.title", u"One Click Maintenance")
    success = _t(props, "final.success", u"Successfully completed")
    none_title = _t(props, "none.title", u" ")
    done = _t(props, "final.done", u"Done:")
    item1 = _t(props, "final.item1", u"Cleanup completed.")
    item2 = _t(props, "final.item2", u"Backup completed.")
    next_steps = _t(props, "final.next", u"Next steps:")
    step1 = _t(props, "final.step1", u"Verify backup folder.")
    step2 = _t(props, "final.step2", u"Close this window.")
    tips = _t(props, "final.tips", u"Tips: timestamp folder is created for each run.")
    try:
        if LAST_BACKUP_FOLDER_NAME:
            tips = tips.replace("yyyy-MM-dd_HH.mm.ss", LAST_BACKUP_FOLDER_NAME)
    except Exception:
        pass
    more = _t(props, "final.more", u"For more content and tutorials, please visit:")
    link_text = _t(props, "final.link", u"Visit Website")
    font_family = _pick_font_family(_FONT_FALLBACKS)
    body = (
        "<div style='font-size:14pt; font-weight:bold;'>%s</div>"
        "<div style='font-size:18pt; font-weight:bold; color:green;'>%s</div>"
        "<div style='margin-top:12px;'><b>%s</b></div>"
        "<div style='margin-left:14px;'>1# %s</div>"
        "<div style='margin-left:14px;'>2# %s</div>"
        "<div style='margin-top:16px;'><b>%s</b></div>"
        "<div style='margin-left:14px; color:red;'>1# %s</div>"
        "<div style='margin-left:14px; color:red;'>2# %s</div>"
        "<div style='margin-top:10px;'>%s</div>"
        "<div style='margin-top:10px;'>%s "
        "<a href='%s' "
        "style='display:inline-block; padding:8px 16px; background:#c3f3c4; color:#0A74DA; "
        "text-decoration:none; border-radius:4px;'>%s</a>"
        "</div>"
    ) % (
        _escape_html(none_title),
        _escape_html(success),
        _escape_html(done),
        _escape_html(item1),
        _escape_html(item2),
        _escape_html(next_steps),
        _escape_html(step1),
        _escape_html(step2),
        _escape_html(tips),
        _escape_html(more),
        _escape_html(WEBSITE_URL),
        _escape_html(link_text),
    )
    html = u"<html><body style='font-family:%s;'>%s</body></html>" % (font_family, body)
    return title, html


# ===========================================
# URL 打开能力：跨平台浏览器打开逻辑
# ===========================================
def _open_url(url):
    if not url:
        return False
    try:
        if webbrowser.open(url, new=2):
            return True
    except Exception:
        pass
    try:
        if Desktop.isDesktopSupported():
            Desktop.getDesktop().browse(URI(url))
            return True
    except Exception:
        pass
    try:
        from java.lang import Runtime
        rt = Runtime.getRuntime()
        if os.name == 'nt':
            rt.exec('cmd /c start "" "%s"' % url.replace('"', '\"'))
        elif sys.platform == 'darwin':
            rt.exec('open "%s"' % url.replace('"', '\"'))
        else:
            rt.exec('xdg-open "%s"' % url.replace('"', '\"'))
        return True
    except Exception:
        pass
    return False


# ===========================================
# 进度文案：按语言选择显示文本
# ===========================================
def _progress_text(key, default_zh, default_en):
    lang = _LANG
    if lang == "en":
        props = _load_translations("en")
        default_text = default_en
    else:
        props = _load_translations("zh")
        default_text = default_zh
    return _t(props, key, default_text)


# ===========================================
# 日志输出：打印并同步到 UI
# ===========================================
def safe_print(msg):
    try:
        print msg
    except Exception:
        try:
            sys.stdout.write(str(msg) + '\n')
        except Exception:
            pass
    try:
        _progress_log(msg)
    except Exception:
        pass


# ===========================================
# 进度窗口：标题栏 + 日志区 + 进度条 + 按钮
# ===========================================
class ProgressDialog(object):
    def __init__(self, start_callback):
        self.start_callback = start_callback
        self.started = False
        self._showing_final = False
        self.dialog = JDialog()
        self.dialog.setModal(False)
        self.dialog.setAlwaysOnTop(True)
        self.dialog.setSize(680, 420)
        self.dialog.setLayout(BorderLayout())

        font_title = _get_ui_font(16)
        font_btn = _get_ui_font(12)
        font_body = _get_ui_font(12)
        font_bar = _get_ui_font(_PROGRESS_BAR_FONT_SIZE)

        header = JPanel()
        header.setLayout(BorderLayout())
        stage_panel = JPanel()
        try:
            stage_panel.setLayout(BoxLayout(stage_panel, BoxLayout.X_AXIS))
        except Exception:
            pass
        self.stage_dialog = JLabel("")
        self.stage_log = JLabel("")
        self.stage_final = JLabel("")
        if font_title is not None:
            try:
                self.stage_dialog.setFont(font_title)
                self.stage_log.setFont(font_title)
                self.stage_final.setFont(font_title)
            except Exception:
                pass
        stage_panel.add(self.stage_dialog)
        try:
            stage_panel.add(Box.createHorizontalStrut(12))
        except Exception:
            pass
        stage_panel.add(self.stage_log)
        try:
            stage_panel.add(Box.createHorizontalStrut(12))
        except Exception:
            pass
        stage_panel.add(self.stage_final)
        header.add(stage_panel, BorderLayout.CENTER)

        btn_panel = JPanel()
        try:
            btn_panel.setLayout(BoxLayout(btn_panel, BoxLayout.X_AXIS))
        except Exception:
            pass
        self.btn_zh = JButton("Chinese")
        self.btn_en = JButton("English")
        if font_btn is not None:
            self.btn_zh.setFont(font_btn)
            self.btn_en.setFont(font_btn)
        btn_panel.add(self.btn_zh)
        btn_panel.add(self.btn_en)
        header.add(btn_panel, BorderLayout.EAST)
        self.dialog.add(header, BorderLayout.NORTH)

        self.area = JTextPane()
        self.area.setEditable(False)
        try:
            self.area.setContentType("text/html")
        except Exception:
            pass
        try:
            class _LinkHandler(HyperlinkListener):
                def hyperlinkUpdate(self, evt):
                    try:
                        if evt.getEventType().toString() == "ACTIVATED":
                            url = None
                            try:
                                url = evt.getURL()
                            except Exception:
                                url = None
                            if url is None:
                                try:
                                    url = evt.getDescription()
                                except Exception:
                                    url = None
                            try:
                                _open_url(str(url))
                            except Exception:
                                pass
                    except Exception:
                        pass
            self.area.addHyperlinkListener(_LinkHandler())
        except Exception:
            pass
        if font_body is not None:
            self.area.setFont(font_body)
        self._lines = []
        self.scroll = JScrollPane(self.area)
        self.dialog.add(self.scroll, BorderLayout.CENTER)

        self.bar = JProgressBar(0, 100)
        self.bar.setStringPainted(True)
        if font_bar is not None:
            self.bar.setFont(font_bar)
        try:
            self.bar.setPreferredSize(Dimension(0, int(_PROGRESS_BAR_HEIGHT)))
        except Exception:
            pass

        self.btn_continue = JButton("")
        self.btn_exit = JButton("")
        if font_btn is not None:
            self.btn_continue.setFont(font_btn)
            self.btn_exit.setFont(font_btn)

        buttons = JPanel()
        try:
            buttons.setLayout(BoxLayout(buttons, BoxLayout.X_AXIS))
        except Exception:
            pass
        try:
            buttons.add(Box.createHorizontalGlue())
        except Exception:
            pass
        buttons.add(self.btn_continue)
        buttons.add(self.btn_exit)

        bottom = JPanel()
        bottom.setLayout(BorderLayout())
        bottom.add(self.bar, BorderLayout.NORTH)
        bottom.add(buttons, BorderLayout.SOUTH)
        self.dialog.add(bottom, BorderLayout.SOUTH)

        def _apply_lang(lang):
            try:
                globals()['_LANG'] = lang
            except Exception:
                pass
            props = _load_translations(lang)
            try:
                self.dialog.setTitle(_t(props, 'window.title', u'One Click Maintenance'))
            except Exception:
                pass
            self.btn_zh.setText(_t(props, 'lang.zh', u'Chinese'))
            self.btn_en.setText(_t(props, 'lang.en', u'English'))
            self.btn_continue.setText(_t(props, 'button.continue', u'Continue'))
            self.btn_exit.setText(_t(props, 'button.exit', u'Exit'))
            try:
                dialog_text = _t(props, 'dialog.title', u'Maintenance Confirm')
                log_text = _t(props, 'log.title', u'Maintenance Log')
                final_text = _t(props, 'final.title', u'AlanDesign Messages')
                if not self.started:
                    self.stage_dialog.setText(dialog_text)
                    self.stage_log.setText(u"")
                    self.stage_final.setText(u"")
                elif self._showing_final:
                    self.stage_dialog.setText(u"")
                    self.stage_log.setText(u"")
                    self.stage_final.setText(final_text)
                else:
                    self.stage_dialog.setText(u"")
                    self.stage_log.setText(log_text)
                    self.stage_final.setText(u"")
            except Exception:
                pass

            if not self.started:
                msg = _t(props, 'dialog.message', u'Maintenance tasks will run. Continue?')
                self.set_text(_format_dialog_message_html(msg), is_html=True)
            elif self._showing_final:
                self.show_final()
            else:
                self._set_html_from_lines(scroll_to_end=True)
            _progress_refresh()

        class _LangHandler(ActionListener):
            def __init__(self, lang):
                self.lang = lang
            def actionPerformed(self, evt):
                _apply_lang(self.lang)

        class _ContinueHandler(ActionListener):
            def actionPerformed(self, evt):
                if self_outer.started:
                    return
                usb_root = None
                try:
                    usb_root = find_usb()
                except Exception:
                    usb_root = None
                if not usb_root:
                    if _USB_FIND_STATUS == "multiple":
                        return
                    try:
                        if _LANG == "en":
                            msg = u"No valid USB drive containing config.txt was found."
                            title = u"USB Not Found"
                        else:
                            msg = u"未找到包含config.txt的有效U盘。"
                            title = u"未找到U盘"
                        JOptionPane.showMessageDialog(self_outer.dialog, msg, title, JOptionPane.WARNING_MESSAGE)
                    except Exception:
                        pass
                    return
                try:
                    globals()["_USB_SELECTED_ROOT"] = usb_root
                except Exception:
                    pass
                self_outer.started = True
                try:
                    self_outer.btn_continue.setEnabled(False)
                except Exception:
                    pass
                try:
                    _save_config_once()
                except Exception:
                    pass
                try:
                    props = _load_translations(_LANG)
                    self_outer.stage_dialog.setText(u"")
                    self_outer.stage_log.setText(_t(props, 'log.title', u'Maintenance Log'))
                    self_outer.stage_final.setText(u"")
                except Exception:
                    pass
                self_outer.append("")
                try:
                    self_outer.start_callback()
                except Exception:
                    pass

        class _ExitHandler(ActionListener):
            def actionPerformed(self, evt):
                try:
                    self_outer.dialog.dispose()
                except Exception:
                    pass
                try:
                    globals()['_PROGRESS_DIALOG'] = None
                except Exception:
                    pass

        self_outer = self
        self.btn_zh.addActionListener(_LangHandler('zh'))
        self.btn_en.addActionListener(_LangHandler('en'))
        self.btn_continue.addActionListener(_ContinueHandler())
        self.btn_exit.addActionListener(_ExitHandler())
        _apply_lang(_LANG)

    def show(self):
        self.dialog.setLocationRelativeTo(None)
        self.dialog.setVisible(True)

    def set_text(self, text, is_html=False):
        def _set():
            try:
                if is_html:
                    self._lines = []
                    self.area.setText(text)
                    self.area.setCaretPosition(0)
                else:
                    self._lines = [text]
                    self._set_html_from_lines()
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def append(self, text):
        def _append():
            try:
                self._lines.append(text)
                self._set_html_from_lines(scroll_to_end=True)
            except Exception:
                pass
        SwingUtilities.invokeLater(_append)

    def set_progress(self, value, text=None):
        def _set():
            try:
                self.bar.setValue(int(value))
                if text is not None:
                    try:
                        self.bar.setString(unicode(text))
                    except Exception:
                        self.bar.setString(str(text))
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def show_final(self):
        self._showing_final = True
        try:
            props = _load_translations(_LANG)
            self.stage_dialog.setText(u"")
            self.stage_log.setText(u"")
            self.stage_final.setText(_t(props, "final.title", u"AlanDesign Messages"))
        except Exception:
            pass
        title, html = _get_final_message_html(_LANG)
        try:
            self.dialog.setTitle(title)
        except Exception:
            pass
        def _set():
            try:
                self.area.setText(html)
                self.area.setCaretPosition(0)
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def _set_html_from_lines(self, scroll_to_end=False):
        font_family = _pick_font_family(_FONT_FALLBACKS)
        body = "<br>".join([_escape_html(l) for l in self._lines])
        html = "<html><body style='font-family:%s; font-size:12pt;'>%s</body></html>" % (font_family, body)
        try:
            self.area.setText(html)
            if scroll_to_end:
                self.area.setCaretPosition(self.area.getDocument().getLength())
            else:
                self.area.setCaretPosition(0)
            try:
                self.scroll.getHorizontalScrollBar().setValue(0)
            except Exception:
                pass
        except Exception:
            pass


# ===========================================
# 进度窗口：初始化
# ===========================================
def _progress_init():
    global _PROGRESS_DIALOG
    if _PROGRESS_DIALOG is None:
        _PROGRESS_DIALOG = ProgressDialog(_start_pipeline_async)
        _PROGRESS_DIALOG.show()


# ===========================================
# 日志输出：追加到窗口日志区
# ===========================================
def _progress_log(msg):
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.append(msg)


# ===========================================
# 进度条：设置数值与文字
# ===========================================
def _progress_set(value, text=None):
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.set_progress(value, text)


# ===========================================
# 进度条（多语言）：保存状态并更新显示
# ===========================================
def _progress_set_i18n(value, key, default_zh, default_en):
    try:
        _PROGRESS_STATE["value"] = value
        _PROGRESS_STATE["key"] = key
        _PROGRESS_STATE["default_zh"] = default_zh
        _PROGRESS_STATE["default_en"] = default_en
    except Exception:
        pass
    _progress_set(value, _progress_text(key, default_zh, default_en))


# ===========================================
# 结果页面：显示最终成功信息
# ===========================================
def _progress_show_final():
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.show_final()


# ===========================================
# 进度条刷新：语言切换时重绘文字
# ===========================================
def _progress_refresh():
    try:
        if _PROGRESS_STATE["value"] is None:
            return
        _progress_set_i18n(
            _PROGRESS_STATE["value"],
            _PROGRESS_STATE["key"],
            _PROGRESS_STATE["default_zh"],
            _PROGRESS_STATE["default_en"],
        )
    except Exception:
        pass


# ===========================================
# 递归删除：用于清理目录与文件
# ===========================================
def _recursive_delete(f):
    try:
        if f is None:
            return
        if f.isDirectory():
            children = f.listFiles()
            if children is None:
                children = []
            for c in children:
                _recursive_delete(c)
            try:
                f.delete()
            except Exception:
                pass
        else:
            try:
                f.delete()
            except Exception:
                pass
    except Exception as e:
        safe_print("_recursive_delete error: %s" % str(e))


# ===========================================
# 根目录：获取 .openpnp2 根路径
# ===========================================
def get_openpnp_root():
    try:
        sd = scripting.getScriptsDirectory().toString()
        parent = os.path.dirname(sd)
        if not parent:
            raise Exception('No scripting parent')
        openpnp_root = parent.replace('/', os.sep)
    except Exception:
        try:
            from java.lang import System
            user_home = System.getProperty('user.home')
        except Exception:
            user_home = os.path.expanduser('~')
        openpnp_root = os.path.join(user_home, '.openpnp2')
    return openpnp_root


# ===========================================
# USB 检测（Windows ctypes）
# 条件：容量范围 + config.txt 存在
# ===========================================
def _config_has_var(config_path):
    if not config_path or not os.path.exists(config_path):
        return False
    try:
        try:
            with io.open(config_path, "r", encoding="utf-8", errors="ignore") as f:
                content = f.read()
        except Exception:
            with io.open(config_path, "r") as f:
                content = f.read()
        pattern = r"^" + re.escape(VAR_NAME) + r"\s+\S+"
        return re.search(pattern, content, re.MULTILINE) is not None
    except Exception:
        return False


def find_usb_with_ctypes():
    try:
        import ctypes
    except Exception:
        return None
    candidates = []
    try:
        for letter in "ABCDEFGHIJKLMNOPQRSTUVWXYZ":
            root = letter + ":\\"
            if not os.path.exists(root):
                continue
            try:
                if ctypes.windll.kernel32.GetDriveTypeW(root) != 2:
                    continue
            except Exception:
                continue
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(root)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测（Java NIO）
# 条件：容量范围 + config.txt 存在
# ===========================================
def find_usb_with_java():
    try:
        import java.io as jio
    except Exception:
        return None
    candidates = []
    try:
        roots = jio.File.listRoots()
        for r in roots:
            root = r.getAbsolutePath()
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(root)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测（macOS /Volumes）
# 条件：容量范围 + config.txt 存在
# ===========================================
def find_usb_on_macos():
    try:
        vols_root = '/Volumes'
        if not os.path.exists(vols_root):
            return None
    except Exception:
        return None
    candidates = []
    try:
        for name in os.listdir(vols_root):
            path = os.path.join(vols_root, name)
            cfg = os.path.join(path, 'config.txt')
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(path if path.endswith(os.sep) else path + os.sep)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测入口：按平台选择检测方式
# ===========================================
def find_usb():
    global _USB_FIND_STATUS
    _USB_FIND_STATUS = None
    candidates = []
    try:
        r = find_usb_with_ctypes()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass
    try:
        from java.lang import System
        osname = System.getProperty('os.name')
    except Exception:
        osname = sys.platform
    if osname and 'mac' in osname.lower():
        try:
            r = find_usb_on_macos()
            if r:
                candidates.extend(list(r))
        except Exception:
            pass
    try:
        r = find_usb_with_java()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass

    unique = []
    seen = set()
    for p in candidates:
        try:
            key = p.lower()
        except Exception:
            key = p
        if key not in seen:
            seen.add(key)
            unique.append(p)

    if not unique:
        _USB_FIND_STATUS = "none"
        return None
    if len(unique) > 1:
        _USB_FIND_STATUS = "multiple"
        try:
            if _LANG == "en":
                msg = u"Multiple USB drives with config.txt found. Please check."
                title = u"Multiple USB Found"
            else:
                msg = u"找到多个含有config.txt的U盘，请检查。"
                title = u"找到多个U盘"
            JOptionPane.showMessageDialog(_PROGRESS_DIALOG.dialog if _PROGRESS_DIALOG else None, msg, title, JOptionPane.WARNING_MESSAGE)
        except Exception:
            pass
        return None
    _USB_FIND_STATUS = "ok"
    return unique[0]


# ===========================================
# 保存配置：调用 OpenPnP saveConfig
# ===========================================
def _save_config_once():
    try:
        if MainFrame is None:
            safe_print("MainFrame API unavailable; skip saveConfig().")
            return False
        frame = MainFrame.get()
        if frame is None:
            safe_print("MainFrame.get() returned None; skip saveConfig().")
            return False
        ok = frame.saveConfig()
        safe_print("frame.saveConfig() called. result=%s" % str(ok))
        return bool(ok)
    except Exception as e:
        safe_print("frame.saveConfig() failed: %s" % str(e))
        try:
            traceback.print_exc()
        except Exception:
            pass
        return False


# ===========================================
# 清理目录：删除 org.openpnp.* 文件夹
# ===========================================
def _clean_org_openpnp_dirs(openpnp_root):
    root_file = File(openpnp_root)
    children = root_file.listFiles() or []
    count = 0
    for ch in children:
        try:
            if ch is None:
                continue
            name = ch.getName()
            if name.startswith('org.openpnp.') and ch.isDirectory():
                safe_print("Deleting org.openpnp.* folder: %s" % ch.getAbsolutePath())
                _recursive_delete(ch)
                count += 1
        except Exception as e:
            safe_print("Failed deleting org.openpnp.* item: %s" % str(e))
    safe_print("Deleted org.openpnp.* folders: %s" % count)


# ===========================================
# 清理日志：删除历史 log（保留 OpenPnP.log）
# ===========================================
def _clean_log_folder(openpnp_root):
    log_dir = os.path.join(openpnp_root, 'log')
    if not os.path.isdir(log_dir):
        safe_print("log folder not found: %s" % log_dir)
        return
    removed = 0
    for name in os.listdir(log_dir):
        path = os.path.join(log_dir, name)
        try:
            if os.path.isdir(path):
                shutil.rmtree(path)
                removed += 1
                safe_print("Removed log subfolder: %s" % path)
                continue
            if os.path.isfile(path) and name != 'OpenPnP.log':
                os.remove(path)
                removed += 1
                safe_print("Removed log file: %s" % path)
        except Exception as e:
            safe_print("Failed removing log item %s: %s" % (path, str(e)))
    safe_print("Removed log items (except OpenPnP.log): %s" % removed)


# ===========================================
# 文件拷贝：单文件复制
# ===========================================
def _copy_file(src, dst):
    parent = os.path.dirname(dst)
    if parent and not os.path.exists(parent):
        os.makedirs(parent)
    shutil.copy2(src, dst)


# ===========================================
# 目录拷贝：整树复制
# ===========================================
def _copy_tree(src, dst):
    if not os.path.exists(src):
        return False
    if os.path.exists(dst):
        shutil.rmtree(dst)
    shutil.copytree(src, dst)
    return True


# ===========================================
# 统计文件：用于日志显示
# ===========================================
def _count_files_in_tree(root_path):
    if not os.path.exists(root_path):
        return 0
    total = 0
    try:
        for _base, _dirs, files in os.walk(root_path):
            total += len(files)
    except Exception:
        return 0
    return total


# ===========================================
# 备份到 U 盘：xml + scripts
# ===========================================
def _backup_to_usb(openpnp_root, usb_root):
    global LAST_BACKUP_FOLDER_NAME
    backup_base = os.path.join(usb_root, 'TornadoSMT_Backups')
    if not os.path.exists(backup_base):
        os.makedirs(backup_base)

    try:
        from java.time import LocalDateTime, DateTimeFormatter
        ts = LocalDateTime.now().format(DateTimeFormatter.ofPattern('yyyy-MM-dd_HH.mm.ss'))
    except Exception:
        import datetime
        ts = datetime.datetime.now().strftime('%Y-%m-%d_%H.%M.%S')

    backup_dir = os.path.join(backup_base, ts)
    if not os.path.exists(backup_dir):
        os.makedirs(backup_dir)
    LAST_BACKUP_FOLDER_NAME = ts

    safe_print("USB backup folder: %s" % backup_dir)

    xml_count = 0
    xml_start = time.time()
    safe_print("Start backing up *.xml files in .openpnp2 folder...")
    for name in os.listdir(openpnp_root):
        src = os.path.join(openpnp_root, name)
        if os.path.isfile(src) and name.lower().endswith('.xml'):
            dst = os.path.join(backup_dir, name)
            _copy_file(src, dst)
            xml_count += 1
            safe_print("Backed up xml: %s" % name)
    xml_sec = time.time() - xml_start
    safe_print("Finished .xml (%s files, %.2f sec)" % (xml_count, xml_sec))

    scripts_root = os.path.join(openpnp_root, 'scripts')
    copied_events = False
    copied_tornado = False

    src_events = os.path.join(scripts_root, 'Events')
    dst_events = os.path.join(backup_dir, 'scripts', 'Events')
    safe_print("Start backing up scripts/Events...")
    events_start = time.time()
    events_files = _count_files_in_tree(src_events)
    try:
        copied_events = _copy_tree(src_events, dst_events)
    except Exception as e:
        safe_print("Failed backing up scripts/Events: %s" % str(e))
    events_sec = time.time() - events_start
    safe_print("Finished scripts/Events (%s files, %.2f sec)" % (events_files if copied_events else 0, events_sec))

    src_tornado = os.path.join(scripts_root, 'TornadoSMT')
    dst_tornado = os.path.join(backup_dir, 'scripts', 'TornadoSMT')
    safe_print("Start backing up scripts/TornadoSMT... please wait...")
    tornado_start = time.time()
    tornado_files = _count_files_in_tree(src_tornado)
    try:
        copied_tornado = _copy_tree(src_tornado, dst_tornado)
    except Exception as e:
        safe_print("Failed backing up scripts/TornadoSMT: %s" % str(e))
    tornado_sec = time.time() - tornado_start
    safe_print("Finished scripts/TornadoSMT (%s files, %.2f sec)" % (tornado_files if copied_tornado else 0, tornado_sec))

    safe_print("Backup summary: xml=%s, events=%s, tornadosmt=%s" % (xml_count, copied_events, copied_tornado))


# ===========================================
# 主流程：保存、清理、备份、结果
# ===========================================
def _run_pipeline():
    try:
        _progress_set_i18n(5, "progress.preparing", u"Preparing...", u"Preparing...")
        openpnp_root = get_openpnp_root()
        safe_print("OpenPnP root: %s" % openpnp_root)

        _progress_set_i18n(20, "progress.saving", u"Saving config...", u"Saving config...")
        safe_print("saveConfig was already triggered right after Continue was clicked.")

        _progress_set_i18n(45, "progress.cleaning", u"Cleaning files...", u"Cleaning files...")
        _clean_org_openpnp_dirs(openpnp_root)
        _clean_log_folder(openpnp_root)

        _progress_set_i18n(75, "progress.backup", u"Backing up to USB...", u"Backing up to USB...")
        usb_root = None
        try:
            usb_root = _USB_SELECTED_ROOT
        except Exception:
            usb_root = None
        if not usb_root:
            if _USB_FIND_STATUS == "multiple":
                return
            safe_print("No matching USB drive found with config.txt. Operation stopped.")
        else:
            safe_print("USB detected: %s" % usb_root)
            _backup_to_usb(openpnp_root, usb_root)
            if RESULT_DELAY_SECONDS > 0:
                safe_print("Waiting %.1f seconds before showing result..." % float(RESULT_DELAY_SECONDS))
                time.sleep(float(RESULT_DELAY_SECONDS))

        _progress_set_i18n(100, "progress.done", u"Done", u"Done")
        _progress_show_final()
    except Exception as e:
        safe_print("Pipeline failed: %s" % str(e))
        try:
            traceback.print_exc()
        except Exception:
            pass


# ===========================================
# 线程启动：后台运行主流程
# ===========================================
def _start_pipeline_async():
    try:
        from java.lang import Thread
    except Exception:
        _run_pipeline()
        return

    class _Runner(java.lang.Runnable):
        def run(self):
            _run_pipeline()

    Thread(_Runner()).start()


# ===========================================
# 入口：初始化窗口并启动流程
# ===========================================
def _run_pipeline_async():
    _progress_init()


_run_pipeline_async()
