#!/usr/bin/env jython

# ===========================================
# 脚本流程概览（当前实现）
# 1) 弹出确认对话框，执行版本检测（低版本提示更新）
# 2) 定位 .openpnp2 根目录，备份并清理旧配置
# 3) 下载配置包并解压到 TornadoSMT_V2
# 4) 合并解压后的 .openpnp2 到 OpenPnP 根目录
# 5) 热加载配置，初始化视觉/机器设置并保存
# 6) 同步 U 盘 config.txt（必要时调整版本与引脚定义）
# 7) 清理临时目录，等待5秒后显示完成提示
# 8) 提示用户按复位按钮并重启 OpenPnP
# ===========================================

# ===========================================
# 依赖导入：Java / Swing / Python 标准库
# 注意：该脚本以 Jython 运行
# ===========================================
from java.io import File, FileOutputStream, BufferedInputStream, FileInputStream, BufferedOutputStream
from java.awt import Desktop, BorderLayout, Font, Dimension
from java.awt.event import ActionListener
from java.net import URI, URL, URLDecoder
from java.util.zip import ZipInputStream
from jarray import zeros
import traceback
import sys
import os
from javax.swing import SwingUtilities, JDialog, JProgressBar, JTextPane, JScrollPane, JPanel, JButton, BoxLayout, JLabel, Box, JOptionPane, Timer
from javax.swing.event import HyperlinkListener
import webbrowser
import subprocess
import io, re, shutil, datetime, time
import java.lang
try:
    from java.nio.file import Files, StandardCopyOption
except Exception:
    Files = None
    StandardCopyOption = None


# ===========================================
# OpenPnP 可选 API：在运行环境可用时启用
# ===========================================
try:
    from org.openpnp.model import Configuration
    from org.openpnp.gui import MainFrame
    from org.pmw.tinylog import Logger
except Exception:
    Configuration = None
    MainFrame = None
    Logger = None
try:
    from org.openpnp import Main
except Exception:
    Main = None


# ===========================================
# 参数区：集中管理可配置项
# 建议：定制时仅修改此处
# ===========================================
_LANG = "zh"  # 默认中文语言
DOWNLOAD_URL = "https://wiki.alandesign.cn/profilepack/tornadosmt_v2-profilepack_v1.3.zip"  # 配置包下载地址
WEBSITE_URL = "https://www.alandesign.org/"# 最后的访问链接
VERSION_CHECK_DELAY_SECONDS = 1.5  # version check delay
MIN_OPENPNP_VERSION = "2.5"  # minimum required version  
# ===========================================
# 运行参数
# ===========================================
DEFAULT_TARGET_DIR = os.path.join(os.path.expanduser("~"), ".openpnp2", "TornadoSMT_V2")  # 默认部署目录
TARGET_DIR = DEFAULT_TARGET_DIR  # 运行时实际使用的目录
VAR_NAME = "endstop.maxz.homing_position"  # 要同步的变量名
_PROGRESS_DIALOG = None
WAIT_SECONDS = 2  # 热加载后等待时间
RESULT_DELAY_SECONDS = 5  # 删除脚本后停留时间（秒）
_PROGRESS_STATE = {
    "value": None,
    "key": None,
    "default_zh": None,
    "default_en": None,
}
_USB_FIND_STATUS = None
_USB_SELECTED_ROOT = None
_PROGRESS_BAR_HEIGHT = 20  # 进度条高度（像素）
_PROGRESS_BAR_FONT_SIZE = 14  # 进度条字号
_DIALOG_MSG_FONT_SIZE_PT = 15  # 确认消息 字号
_DIALOG_MSG_COLOR = "#D00000"  # 确认消息 颜色

# ===========================================
# URL 打开能力：跨平台浏览器打开逻辑
# 兼容 Windows / macOS / Linux
# ===========================================
def _parse_version_parts(text):
    if not text:
        return None
    try:
        parts = re.findall(r"\d+", str(text))
        if not parts:
            return None
        return [int(p) for p in parts]
    except Exception:
        return None

def _version_gte(current_text, min_text):
    cur = _parse_version_parts(current_text)
    req = _parse_version_parts(min_text)
    if not cur or not req:
        return False
    max_len = max(len(cur), len(req))
    cur = cur + [0] * (max_len - len(cur))
    req = req + [0] * (max_len - len(req))
    return cur >= req

def _get_openpnp_version():
    if Main is None:
        return None
    try:
        return Main.getVersionString()
    except Exception:
        return None

def _open_url(url):
    if not url:
        return False

    # 1) 尝试 webbrowser
    try:
        if webbrowser.open(url, new=2):
            return True
    except Exception:
        pass

    # 2) 尝试 Java Desktop
    try:
        if Desktop.isDesktopSupported():
            try:
                desktop = Desktop.getDesktop()
                try:
                    Action = Desktop.Action
                    if not desktop.isSupported(Action.BROWSE):
                        raise Exception("Desktop.BROWSE not supported")
                except Exception:
                    pass
                desktop.browse(URI(url))
                return True
            except Exception:
                pass
    except Exception:
        pass

    # 3) Runtime.exec 回退
    try:
        from java.lang import Runtime
        rt = Runtime.getRuntime()
        if os.name == 'nt':
            cmd = 'cmd /c start "" "%s"' % url.replace('"', '\\"')
            rt.exec(cmd)
        elif sys.platform == 'darwin':
            cmd = 'open "%s"' % url.replace('"', '\\"')
            rt.exec(cmd)
        else:
            cmd = 'xdg-open "%s"' % url.replace('"', '\\"')
            rt.exec(cmd)
        return True
    except Exception:
        pass

    # 4) 额外回退（Windows）
    if os.name == 'nt':
        try:
            subprocess.Popen(['rundll32', 'url.dll,FileProtocolHandler', url], shell=False)
            return True
        except Exception:
            pass
        try:
            subprocess.Popen(['powershell', '-NoProfile', '-Command', 'Start-Process', '"%s"' % url], shell=False)
            return True
        except Exception:
            pass

    # 5) 最后再试系统级命令
    try:
        if os.name == 'nt':
            subprocess.Popen('start "" "%s"' % url.replace('"', '\\"'), shell=True)
            return True
        elif sys.platform == 'darwin':
            subprocess.Popen(['open', url])
            return True
        else:
            subprocess.Popen(['xdg-open', url])
            return True
    except Exception:
        pass

    return False

def _format_path_for_log(path):
    if path is None:
        return None
    try:
        root = get_openpnp_root()
    except Exception:
        root = None
    try:
        if root and path.startswith(root):
            rel = path[len(root):]
            if rel.startswith(os.sep):
                rel = rel[1:]
            return "root" + (os.sep + rel if rel else "")
    except Exception:
        pass
    return path

def _format_java_file_path_for_log(java_file):
    try:
        abs_path = java_file.getAbsolutePath()
    except Exception:
        abs_path = None
    return _format_path_for_log(abs_path)

# ===========================================
# 内置多语言文本
# 作用：用于窗口标题、提示、日志与结果文案
# ===========================================
INLINE_I18N_ZH = {
    "button.continue": u"继续",
    "button.exit": u"退出",
    "dialog.message": u"后续操作，将重置 .openpnp2文件夹 和 U盘 中的配置文件，至初始状态。\n\n是否继续？",
    "dialog.title": u"一键部署 确认",
    "log.title": u"一键部署 日志",
    "final.done": u"已完成：",
    "final.item1": u".openpnp2 文件夹部署完毕。",
    "final.item2": u"U盘 config.txt 更新完成。",
    "final.next": u"下一步：",
    "final.step1": u"请按住贴片机底部的复位按钮持续 3 秒钟。",
    "final.step2": u"手动关闭并重启 OpenPnP。",
    "final.step3": u"点击右下角 退出 按钮关闭此窗口。",
    "final.success": u"一键部署，成功完成！！",
    "final.tips": u"提示：复位按钮位于贴片机电源开关附近。",
    "final.more": u"后续更多内容和教程，请访问：",
    "final.link": u" AlanDesign.org ",
    "final.title": u"一键部署 完成",
    "lang.en": u"English",
    "lang.zh": u"中文",
    #"none.title": u" ",
    "progress.done": u"完成",
    "progress.downloading": u"下载中...",
    "progress.merging": u"合并配置中...",
    "progress.preparing": u"准备中...",
    "progress.reloading": u"重载配置中...",
    "progress.syncing": u"同步 U盘...",
    "progress.unzip_done": u"解压完成",
    "progress.unzipping": u"解压中...",
    "window.title": u"一键部署 - by AlanDesign",
}

INLINE_I18N_EN = {
     
    "dialog.message": u"This script will Reset all configurations in .openpnp2 folder and USB dirver.\n\nContinue?",
    "final.link": u" AlanDesign.org ",
    "lang.en": u"English",
    "lang.zh": u"中文",
    
}


# ===========================================
# 字体回退：保证中文可读
# ===========================================
_FONT_FALLBACKS = [
    "Microsoft YaHei",
    "PingFang SC",
    "Hiragino Sans GB",
    "Noto Sans CJK SC",
    "WenQuanYi Micro Hei",
    "Dialog",
]


# ===========================================
# 翻译字典：按语言返回文案集合
# ===========================================
def _load_translations(lang):
    if lang == "en":
        return INLINE_I18N_EN
    return INLINE_I18N_ZH

# ===========================================
# 字典取值：key 不存在时回退默认文本
# ===========================================
def _t(props, key, default_text):
    try:
        return props.get(key, default_text)
    except Exception:
        return default_text


# ===========================================
# 字体选择：挑选可显示中文的字体
# ===========================================
def _pick_font_family(fallbacks):
    sample = u"中文ABC"
    for name in fallbacks:
        try:
            f = Font(name, Font.PLAIN, 12)
            try:
                if f.canDisplayUpTo(sample) == -1:
                    return name
            except Exception:
                try:
                    if f.getFamily() == name:
                        return name
                except Exception:
                    pass
        except Exception:
            pass
    return "Dialog"


# ===========================================
# 字体对象：按字号构造 Font
# ===========================================
def _get_ui_font(size):
    try:
        name = _pick_font_family(_FONT_FALLBACKS)
        return Font(name, Font.PLAIN, int(size))
    except Exception:
        return None


# ===========================================
# HTML 转义：避免文本破坏 HTML 结构
# ===========================================
def _escape_html(text):
    if text is None:
        return ""
    try:
        s = unicode(text)
    except Exception:
        try:
            s = str(text)
        except Exception:
            return ""
    s = s.replace("&", "&amp;")
    s = s.replace("<", "&lt;")
    s = s.replace(">", "&gt;")
    s = s.replace('"', "&quot;")
    s = s.replace("'", "&#39;")
    return s


# ===========================================
# 对话框提示：生成带样式的 HTML
# ===========================================
def _format_dialog_message_html(text):
    font_family = _pick_font_family(_FONT_FALLBACKS)
    safe_text = _escape_html(text)
    safe_text = safe_text.replace("\n", "<br/>")
    html = (
        u"<html><body style='font-family:%s; font-size:%dpt; color:%s;'>%s</body></html>"
        % (font_family, int(_DIALOG_MSG_FONT_SIZE_PT), _DIALOG_MSG_COLOR, safe_text)
    )
    return html


# ===========================================
# 完成页面：拼接最终结果 HTML
# ===========================================
def _get_final_message_html(lang):
    props = _load_translations(lang)
    title = _t(props, "window.title", u"One Click Deployment -by AlanDesign")
    success = _t(props, "final.success", u"Successfully Completed !!")
    none_title = _t(props, "none.title", u" ")
    done = _t(props, "final.done", u"Done:")
    item1 = _t(props, "final.item1", u".openpnp2 configuration files has been deployed.")
    item2 = _t(props, "final.item2", u"config.txt in the USB drive has been updated.")
    next_steps = _t(props, "final.next", u"Next steps:")
    step1 = _t(props, "final.step1", u"Press and hold the Reset Button for more than 3 seconds.")
    step2 = _t(props, "final.step2", u"Manually Close and Restart OpenPnP.")
    tips = _t(props, "final.tips", u"Tips: Reset Button is under and nearby the Power Switch.")
    more = _t(props, "final.more", u"For more content and tutorials, please visit:")
    link_text = _t(props, "final.link", u" AlanDesign.org ")
    font_family = _pick_font_family(_FONT_FALLBACKS)
    body = (
        "<div style='font-size:14pt; font-weight:bold;'>%s</div>"
        "<div style='font-size:18pt; font-weight:bold; color:green;'>%s</div>"
        "<div style='margin-top:12px;'><b>%s</b></div>"
        "<div style='margin-left:14px;'>1# %s</div>"
        "<div style='margin-left:14px;'>2# %s</div>"
        "<div style='margin-top:16px;'><b>%s</b></div>"
        "<div style='margin-left:14px; color:red;'>1# %s</div>"
        "<div style='margin-left:14px; color:red;'>2# %s</div>"
        "<div style='margin-top:10px;'>%s</div>"
        "<div style='margin-top:10px;'>%s "
        "<a href='%s' "
        "style='display:inline-block; padding:8px 16px; background:#c3f3c4; color:#0A74DA; "
        "text-decoration:none; border-radius:4px;'>%s</a>"
        "</div>"
    ) % (
        _escape_html(none_title),
        _escape_html(success),
        _escape_html(done),
        _escape_html(item1),
        _escape_html(item2),
        _escape_html(next_steps),
        _escape_html(step1),
        _escape_html(step2),
        _escape_html(tips),
        _escape_html(more),
        _escape_html(WEBSITE_URL),
        _escape_html(link_text),
    )
    
    html = u"<html><body style='font-family:%s;'>%s</body></html>" % (font_family, body)
    return title, html


# ===========================================
# 进度文案：按语言选择显示文本
# ===========================================
def _progress_text(key, default_zh, default_en):
    lang = _LANG
    if lang == "en":
        props = _load_translations("en")
        default_text = default_en
    else:
        props = _load_translations("zh")
        default_text = default_zh
    return _t(props, key, default_text)


def safe_print(msg):
    try:
        print msg
    except Exception:
        try:
            sys.stdout.write(str(msg) + '\n')
        except Exception:
            pass
    try:
        _progress_log(msg)
    except Exception:
        pass


def safe_log(stage, msg):
    try:
        safe_print("[%s] %s" % (stage, msg))
    except Exception:
        pass


# ===========================================
# 进度窗口：标题栏 + 日志区 + 进度条 + 按钮
# ===========================================
class ProgressDialog(object):
    def __init__(self, start_callback):
        self.start_callback = start_callback
        self.started = False
        self._showing_final = False
        self._version_checked = False
        self.dialog = JDialog()
        self.dialog.setModal(False)
        self.dialog.setAlwaysOnTop(True)
        self.dialog.setSize(680, 420)
        self.dialog.setLayout(BorderLayout())

        font_title = _get_ui_font(16)
        font_btn = _get_ui_font(12)
        font_body = _get_ui_font(12)
        font_bar = _get_ui_font(_PROGRESS_BAR_FONT_SIZE)

        header = JPanel()
        header.setLayout(BorderLayout())
        
        stage_panel = JPanel()
        try:
            stage_panel.setLayout(BoxLayout(stage_panel, BoxLayout.X_AXIS))
        except Exception:
            pass
        self.stage_dialog = JLabel("")
        self.stage_log = JLabel("")
        self.stage_final = JLabel("")
        if font_title is not None:
            try:
                self.stage_dialog.setFont(font_title)
                self.stage_log.setFont(font_title)
                self.stage_final.setFont(font_title)
            except Exception:
                pass
        stage_panel.add(self.stage_dialog)
        try:
            stage_panel.add(Box.createHorizontalStrut(12))
        except Exception:
            pass
        stage_panel.add(self.stage_log)
        try:
            stage_panel.add(Box.createHorizontalStrut(12))
        except Exception:
            pass
        stage_panel.add(self.stage_final)
        header.add(stage_panel, BorderLayout.CENTER)

        btn_panel = JPanel()
        try:
            btn_panel.setLayout(BoxLayout(btn_panel, BoxLayout.X_AXIS))
        except Exception:
            pass
        self.btn_zh = JButton("Chinese")
        self.btn_en = JButton("English")
        if font_btn is not None:
            self.btn_zh.setFont(font_btn)
            self.btn_en.setFont(font_btn)
        btn_panel.add(self.btn_zh)
        btn_panel.add(self.btn_en)
        header.add(btn_panel, BorderLayout.EAST)
        self.dialog.add(header, BorderLayout.NORTH)

        self.area = JTextPane()
        self.area.setEditable(False)
        try:
            self.area.setContentType("text/html")
        except Exception:
            pass
        # 让 HTML 链接可点击并打开浏览器（使用跨平台打开逻辑）
        try:
            class _LinkHandler(HyperlinkListener):
                def hyperlinkUpdate(self, evt):
                    try:
                        if evt.getEventType().toString() == "ACTIVATED":
                            url = None
                            try:
                                url = evt.getURL()
                            except Exception:
                                url = None
                            if url is None:
                                try:
                                    url = evt.getDescription()
                                except Exception:
                                    url = None
                            try:
                                _open_url(str(url))
                            except Exception:
                                pass
                    except Exception:
                        pass
            self.area.addHyperlinkListener(_LinkHandler())
        except Exception:
            pass
        if font_body is not None:
            self.area.setFont(font_body)
        self._lines = []
        self.scroll = JScrollPane(self.area)
        self.dialog.add(self.scroll, BorderLayout.CENTER)

        self.bar = JProgressBar(0, 100)
        self.bar.setStringPainted(True)
        if font_bar is not None:
            self.bar.setFont(font_bar)
        try:
            self.bar.setPreferredSize(Dimension(0, int(_PROGRESS_BAR_HEIGHT)))
        except Exception:
            pass

        self.btn_continue = JButton("")
        self.btn_exit = JButton("")
        if font_btn is not None:
            self.btn_continue.setFont(font_btn)
            self.btn_exit.setFont(font_btn)
        try:
            self.btn_continue.setEnabled(False)
        except Exception:
            pass

        buttons = JPanel()
        try:
            buttons.setLayout(BoxLayout(buttons, BoxLayout.X_AXIS))
        except Exception:
            pass
        try:
            buttons.add(Box.createHorizontalGlue())
        except Exception:
            pass
        buttons.add(self.btn_continue)
        buttons.add(self.btn_exit)

        bottom = JPanel()
        bottom.setLayout(BorderLayout())
        bottom.add(self.bar, BorderLayout.NORTH)
        bottom.add(buttons, BorderLayout.SOUTH)
        self.dialog.add(bottom, BorderLayout.SOUTH)

        def _apply_lang(lang):
            try:
                globals()["_LANG"] = lang
            except Exception:
                pass
            props = _load_translations(lang)
            if lang == "en":
                title_text = _t(props, "window.title", u"One Click Deployment -by AlanDesign")
            else:
                title_text = _t(props, "window.title", u"One Click Deployment -by AlanDesign")
            try:
                self.dialog.setTitle(title_text)
            except Exception:
                pass
            self.btn_zh.setText(_t(props, "lang.zh", u"Chinese"))
            self.btn_en.setText(_t(props, "lang.en", u"English"))
            self.btn_continue.setText(_t(props, "button.continue", u"Continue"))
            self.btn_exit.setText(_t(props, "button.exit", u"Exit"))
            
            try:
                dialog_text = _t(props, "dialog.title", u"Deployment WARNING")
                log_text = _t(props, "log.title", u"Deployment Log")
                final_text = _t(props, "final.title", u"AlanDesign Messages")
                
                if not self.started:
                    self.stage_dialog.setText(dialog_text)
                    self.stage_log.setText(u"")
                    self.stage_final.setText(u"")
                elif self._showing_final:
                    self.stage_dialog.setText(u"")
                    self.stage_log.setText(u"")
                    self.stage_final.setText(final_text)
                else:
                    self.stage_dialog.setText(u"")
                    self.stage_log.setText(log_text)
                    self.stage_final.setText(u"")
            except Exception:
                pass
            if not self.started:
                msg = _t(props, "dialog.message", u"This script will Reset all configurations in .openpnp2 folder and USB dirver. Continue?")
                self.set_text(_format_dialog_message_html(msg), is_html=True)
            elif self._showing_final:
                self.show_final()
            else:
                self._set_html_from_lines(scroll_to_end=True)
            _progress_refresh()

        class _LangHandler(ActionListener):
            def __init__(self, lang):
                self.lang = lang

            def actionPerformed(self, evt):
                _apply_lang(self.lang)

        class _ContinueHandler(ActionListener):
            def actionPerformed(self, evt):
                if self_outer.started:
                    return
                usb_root = None
                try:
                    usb_root = find_usb()
                except Exception:
                    usb_root = None
                if not usb_root:
                    if _USB_FIND_STATUS == "multiple":
                        return
                    try:
                        if _LANG == "en":
                            msg = u"No valid USB drive containing config.txt was found."
                            title = u"USB Not Found"
                        else:
                            msg = u"未找到包含config.txt的有效U盘。"
                            title = u"U盘未找到"
                        JOptionPane.showMessageDialog(self_outer.dialog, msg, title, JOptionPane.WARNING_MESSAGE)
                    except Exception:
                        pass
                    return
                try:
                    globals()["_USB_SELECTED_ROOT"] = usb_root
                except Exception:
                    pass
                self_outer.started = True
                try:
                    self_outer.btn_continue.setEnabled(False)
                except Exception:
                    pass
                try:
                    props = _load_translations(_LANG)
                    self_outer.stage_dialog.setText(u"")
                    self_outer.stage_log.setText(_t(props, "log.title", u"Deployment Log"))
                    self_outer.stage_final.setText(u"")
                except Exception:
                    pass
                self_outer.append("")
                try:
                    self_outer.start_callback()
                except Exception:
                    pass

        class _ExitHandler(ActionListener):
            def actionPerformed(self, evt):
                try:
                    self_outer.dialog.dispose()
                except Exception:
                    pass
                try:
                    globals()["_PROGRESS_DIALOG"] = None
                except Exception:
                    pass

        self_outer = self
        self.btn_zh.addActionListener(_LangHandler("zh"))
        self.btn_en.addActionListener(_LangHandler("en"))
        self.btn_continue.addActionListener(_ContinueHandler())
        self.btn_exit.addActionListener(_ExitHandler())
        _apply_lang(_LANG)

    def _schedule_version_check(self):
        if self._version_checked:
            return
        try:
            delay_s = float(VERSION_CHECK_DELAY_SECONDS)
        except Exception:
            delay_s = 0
        if delay_s <= 0:
            delay_s = 0.0
        try:
            delay_ms = int(delay_s * 1000)
        except Exception:
            delay_ms = 0
        if delay_ms < 0:
            delay_ms = 0

        def _run(evt=None):
            try:
                self._version_checked = True
                version = _get_openpnp_version()
                ok = _version_gte(version, MIN_OPENPNP_VERSION)
                if ok:
                    try:
                        if not self.started:
                            self.btn_continue.setEnabled(True)
                    except Exception:
                        pass
                    return
                try:
                    JOptionPane.showMessageDialog(self.dialog, u"软件版本需要更新。", u"提示", JOptionPane.WARNING_MESSAGE)
                except Exception:
                    pass
                url = "https://www.alandesign.org/slides/slide/%E4%B8%8B%E8%BD%BD%E5%B9%B6%E5%AE%89%E8%A3%85openpnp-24?fullscreen=1"
                msg = u"OpenPnP软件需要更新，跳转链接：<a href='%s'>%s</a>" % (url, url)
                html = u"<html><body>%s</body></html>" % msg
                self.show_custom_result(u"版本更新", html)
            except Exception:
                pass

        try:
            t = Timer(delay_ms, _run)
            try:
                t.setRepeats(False)
            except Exception:
                pass
            t.start()
        except Exception:
            pass

    def show(self):
        self.dialog.setLocationRelativeTo(None)
        self.dialog.setVisible(True)
        try:
            self._schedule_version_check()
        except Exception:
            pass

    def set_text(self, text, is_html=False):
        def _set():
            try:
                if is_html:
                    self._lines = []
                    self.area.setText(text)
                    self.area.setCaretPosition(0)
                else:
                    self._lines = [text]
                    self._set_html_from_lines()
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def append(self, text):
        def _append():
            try:
                self._lines.append(text)
                self._set_html_from_lines(scroll_to_end=True)
            except Exception:
                pass
        SwingUtilities.invokeLater(_append)

    def set_progress(self, value, text=None):
        def _set():
            try:
                self.bar.setValue(int(value))
                if text is not None:
                    try:
                        self.bar.setString(unicode(text))
                    except Exception:
                        self.bar.setString(str(text))
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def show_final(self):
        self._showing_final = True
        try:
            props = _load_translations(_LANG)
            self.stage_dialog.setText(u"")
            self.stage_log.setText(u"")
            self.stage_final.setText(_t(props, "final.title", u"AlanDesign Messages"))
        except Exception:
            pass
        title, html = _get_final_message_html(_LANG)
        try:
            self.dialog.setTitle(title)
        except Exception:
            pass
        def _set():
            try:
                self.area.setText(html)
                self.area.setCaretPosition(0)
            except Exception:
                pass
        SwingUtilities.invokeLater(_set)

    def _set_html_from_lines(self, scroll_to_end=False):
        font_family = _pick_font_family(_FONT_FALLBACKS)
        body = "<br>".join([_escape_html(l) for l in self._lines])
        html = "<html><body style='font-family:%s; font-size:12pt;'>%s</body></html>" % (font_family, body)
        try:
            self.area.setText(html)
            if scroll_to_end:
                self.area.setCaretPosition(self.area.getDocument().getLength())
            else:
                self.area.setCaretPosition(0)
            try:
                self.scroll.getHorizontalScrollBar().setValue(0)
            except Exception:
                pass
        except Exception:
            pass


# ===========================================
# 进度窗口：初始化
# ===========================================
def _progress_init():
    global _PROGRESS_DIALOG
    if _PROGRESS_DIALOG is None:
        _PROGRESS_DIALOG = ProgressDialog(_start_pipeline_async)
        _PROGRESS_DIALOG.show()


# ===========================================
# 日志输出：追加到窗口日志区
# ===========================================
def _progress_log(msg):
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.append(msg)


# ===========================================
# 进度条：设置数值与文字
# ===========================================
def _progress_set(value, text=None):
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.set_progress(value, text)


# ===========================================
# 进度条（多语言）：保存状态并更新显示
# ===========================================
def _progress_set_i18n(value, key, default_zh, default_en):
    try:
        _PROGRESS_STATE["value"] = value
        _PROGRESS_STATE["key"] = key
        _PROGRESS_STATE["default_zh"] = default_zh
        _PROGRESS_STATE["default_en"] = default_en
    except Exception:
        pass
    text = _progress_text(key, default_zh, default_en)
    _progress_set(value, text)


# ===========================================
# 结果页面：显示最终成功信息
# ===========================================
def _progress_show_final():
    if _PROGRESS_DIALOG is not None:
        _PROGRESS_DIALOG.show_final()


def _delete_self_script():
    try:
        sd = scripting.getScriptsDirectory().toString()
        path = os.path.join(sd, 'TornadoSMT', 'One-Click-Deployment.py')
    except Exception:
        path = None
    display_path = _format_path_for_log(path)
    if not path:
        safe_print("Delete One-Click-Deployment.py skipped: scripting.getScriptsDirectory() unavailable.")
        return False
    try:
        f = File(path)
        if f.exists():
            ok = f.delete()
            safe_print("Delete One-Click-Deployment.py via java.io.File: %s (ok=%s)" % (display_path, ok))
            if ok:
                return True
        else:
            safe_print("Delete One-Click-Deployment.py skipped: file not found at %s" % display_path)
    except Exception as e:
        safe_print("Delete One-Click-Deployment.py via java.io.File failed: %s" % str(e))
    try:
        if os.path.exists(path):
            os.remove(path)
            safe_print("Delete One-Click-Deployment.py via os.remove: %s" % display_path)
            return True
    except Exception:
        pass
    return False


# ===========================================
# 进度条刷新：语言切换时重绘文字
# ===========================================
def _progress_refresh():
    try:
        if _PROGRESS_STATE["value"] is None:
            return
        _progress_set_i18n(
            _PROGRESS_STATE["value"],
            _PROGRESS_STATE["key"],
            _PROGRESS_STATE["default_zh"],
            _PROGRESS_STATE["default_en"],
        )
    except Exception:
        pass


# ===========================================
# 目录创建：封装 mkdirs 并打印日志
# ===========================================
def do_mkdirs_file(java_file):
    try:
        if java_file.exists():
            return True
        res = java_file.mkdirs()
        display_path = _format_java_file_path_for_log(java_file)
        safe_print("Created directory: %s" % display_path)
        return res
    except Exception as e:
        safe_print("do_mkdirs_file error: %s" % str(e))
        return False

def _recursive_copy(src_file, dest_file):
    try:
        if src_file.isDirectory():
            if not dest_file.exists():
                dest_file.mkdirs()
            children = src_file.listFiles()
            if children is None:
                children = []
            for c in children:
                _recursive_copy(c, File(dest_file, c.getName()))
        else:
            in_stream = None
            out_stream = None
            temp_file = None
            try:
                in_stream = FileInputStream(src_file)
                parent = dest_file.getParentFile()
                if parent is not None and not parent.exists():
                    do_mkdirs_file(parent)
                temp_path = dest_file.getAbsolutePath() + '.tmp'
                temp_file = File(temp_path)
                out_stream = FileOutputStream(temp_file)
                buf = zeros(4096, 'b')
                read = in_stream.read(buf)
                while read != -1:
                    if read > 0:
                        out_stream.write(buf, 0, read)
                    read = in_stream.read(buf)
                try:
                    out_stream.flush()
                except Exception:
                    pass
            finally:
                try:
                    if in_stream is not None:
                        in_stream.close()
                except Exception:
                    pass
                try:
                    if out_stream is not None:
                        out_stream.close()
                except Exception:
                    pass
            try:
                moved = False
                if Files is not None:
                    try:
                        Files.move(temp_file.toPath(), dest_file.toPath(), StandardCopyOption.REPLACE_EXISTING)
                        moved = True
                    except Exception:
                        moved = False
                if not moved:
                    try:
                        if dest_file.exists():
                            _recursive_delete(dest_file)
                    except Exception:
                        pass
                    try:
                        moved = temp_file.renameTo(dest_file)
                    except Exception:
                        moved = False
                
            except Exception as e:
                safe_print("Failed to move temp file into place: %s" % str(e))
                try:
                    traceback.print_exc()
                except Exception:
                    pass
    except Exception as e:
        safe_print("_recursive_copy error: %s" % str(e))
        try:
            traceback.print_exc()
        except Exception:
            pass
        raise


# ===========================================
# 递归删除：用于清理目录与文件
# ===========================================
def _recursive_delete(f):
    try:
        if f is None:
            return
        if f.isDirectory():
            children = f.listFiles()
            if children is None:
                children = []
            for c in children:
                _recursive_delete(c)
            try:
                f.delete()
            except Exception:
                pass
        else:
            try:
                f.delete()
            except Exception:
                pass
    except Exception as e:
        safe_print("_recursive_delete error: %s" % str(e))
        try:
            traceback.print_exc()
        except Exception:
            pass


# ===========================================
# URL 文件名：从 URL 解析文件名
# ===========================================
def get_filename_from_url(url):
    path = url.split('/')[-1]
    return URLDecoder.decode(path, "UTF-8")


# ===========================================
# 根目录：获取 .openpnp2 根路径
# ===========================================
def get_openpnp_root():
    try:
        sd = scripting.getScriptsDirectory().toString()
        parent = os.path.dirname(sd)
        if not parent:
            raise Exception('No scripting parent')
        openpnp_root = parent.replace('/', os.sep)
    except Exception:
        try:
            from java.lang import System
            user_home = System.getProperty('user.home')
        except Exception:
            user_home = os.path.expanduser('~')
        openpnp_root = os.path.join(user_home, '.openpnp2')
    return openpnp_root


# ===========================================
# 目标目录：备份旧配置并清理环境
# ===========================================
def resolve_target_dir():
    safe_log("Cleanup", "Starting")
    try:
        sd = scripting.getScriptsDirectory().toString()
        parent = os.path.dirname(sd)
        if not parent:
            raise Exception('No scripting parent')
        openpnp_root = parent.replace('/', os.sep)
    except Exception:
        try:
            from java.lang import System
            user_home = System.getProperty('user.home')
        except Exception:
            user_home = os.path.expanduser('~')
        openpnp_root = os.path.join(user_home, '.openpnp2')

    try:
        rf = File(openpnp_root)
        do_mkdirs_file(rf)
    except Exception:
        safe_log("Cleanup", "Failed: ensure root")

    try:
        fail_org = 0
        children = rf.listFiles() or []
        for ch in children:
            try:
                if ch is None:
                    continue
                name = ch.getName()
                if name.startswith('org.openpnp.'):
                    _recursive_delete(ch)
            except Exception:
                fail_org += 1
        if fail_org:
            safe_log("Cleanup", "Failed: remove org.openpnp.* (%s items)" % fail_org)
    except Exception:
        safe_log("Cleanup", "Failed: scan org.openpnp.*")

    try:
        v2_dir = os.path.join(openpnp_root, 'TornadoSMT_V2')
        v2_file = File(v2_dir)
        if v2_file.exists():
            fail_v2 = 0
            try:
                for c in v2_file.listFiles() or []:
                    try:
                        _recursive_delete(c)
                    except Exception:
                        fail_v2 += 1
            except Exception:
                safe_log("Cleanup", "Failed: enumerate TornadoSMT_V2")
            if fail_v2:
                safe_log("Cleanup", "Failed: clear TornadoSMT_V2 (%s items)" % fail_v2)
        else:
            try:
                do_mkdirs_file(File(v2_dir))
            except Exception:
                safe_log("Cleanup", "Failed: create TornadoSMT_V2")
    except Exception:
        safe_log("Cleanup", "Failed: handle TornadoSMT_V2")

    backups_dir = os.path.join(openpnp_root, 'TornadoSMT_Backups')
    try:
        do_mkdirs_file(File(backups_dir))
    except Exception:
        pass

    try:
        try:
            from java.time import LocalDateTime, DateTimeFormatter
            fmt = DateTimeFormatter.ofPattern('yyyy-MM-dd_HH.mm.ss')
            ts = LocalDateTime.now().format(fmt)
        except Exception:
            import datetime
            ts = datetime.datetime.now().strftime('%Y-%m-%d_%H.%M.%S')

        timestamp_folder = os.path.join(backups_dir, ts)
        tdir = File(timestamp_folder)
        do_mkdirs_file(tdir)

        backup_dot_openpnp = File(timestamp_folder, '.openpnp2')
        usb_config = File(timestamp_folder, 'USB Config')
        do_mkdirs_file(backup_dot_openpnp)
        do_mkdirs_file(usb_config)
        try:
            globals()['USB_CONFIG_DIR'] = usb_config.getAbsolutePath()
        except Exception:
            pass

        current_children = File(openpnp_root).listFiles() or []
        backup_fail = 0
        for item in current_children:
            try:
                if item is None:
                    continue
                nm = item.getName()
                if nm == 'TornadoSMT_Backups' or nm == 'TornadoSMT_V2' or nm == tdir.getName():
                    continue
                dest = File(backup_dot_openpnp, nm)
                _recursive_copy(item, dest)
            except Exception:
                backup_fail += 1
        if backup_fail:
            safe_log("Cleanup", "Failed: backup .openpnp2 (%s items)" % backup_fail)
    except Exception:
        safe_log("Cleanup", "Failed: backup block")

    try:
        remaining = File(openpnp_root).listFiles() or []
        remaining_fail = 0
        for rem in remaining:
            try:
                if rem is None:
                    continue
                name = rem.getName()
                if name == 'TornadoSMT_Backups' or name == 'TornadoSMT_V2' or name == 'scripts':
                    continue
                if name.startswith('TornadoSMT'):
                    continue
                if name == 'log' and rem.isDirectory():
                    try:
                        log_children = rem.listFiles() or []
                        log_fail = 0
                        for lf in log_children:
                            try:
                                if lf is None:
                                    continue
                                if lf.isFile() and lf.getName().lower().endswith('.log') and lf.getName() != 'OpenPnP.log':
                                    _recursive_delete(lf)
                            except Exception:
                                log_fail += 1
                    except Exception:
                        log_fail = log_fail + 1
                    if log_fail:
                        safe_log("Cleanup", "Failed: clear log files (%s items)" % log_fail)
                    continue
                _recursive_delete(rem)
            except Exception:
                remaining_fail += 1
        if remaining_fail:
            safe_log("Cleanup", "Failed: remove remaining items (%s items)" % remaining_fail)
    except Exception:
        safe_log("Cleanup", "Failed: remove remaining items block")

    try:
        scripts_dir = os.path.join(openpnp_root, 'scripts')
        do_mkdirs_file(File(scripts_dir))
        tornado_scripts = os.path.join(scripts_dir, 'TornadoSMT')
        do_mkdirs_file(File(tornado_scripts))
        examples_scripts = os.path.join(scripts_dir, 'Examples')
        do_mkdirs_file(File(examples_scripts))
    except Exception:
        safe_log("Cleanup", "Failed: create scripts dirs")

    try:
        scripts_file = File(scripts_dir)
        events_dir = File(scripts_dir, 'Events')
        examples_dir = File(scripts_dir, 'Examples')
        tornado_dir = File(scripts_dir, 'TornadoSMT')
        do_mkdirs_file(events_dir)
        do_mkdirs_file(examples_dir)
        do_mkdirs_file(tornado_dir)

        js_dir = File(examples_dir, 'JavaScript')
        py_dir = File(examples_dir, 'Python')
        do_mkdirs_file(js_dir)
        do_mkdirs_file(py_dir)

        children = scripts_file.listFiles() or []
        scripts_fail = 0
        for ch in children:
            try:
                if ch is None:
                    continue
                name = ch.getName()
                if name == 'TornadoSMT':
                    continue
                if name == 'Events' and ch.isDirectory():
                    try:
                        ev_children = ch.listFiles() or []
                        for ev in ev_children:
                            try:
                                _recursive_delete(ev)
                            except Exception:
                                scripts_fail += 1
                    except Exception:
                        scripts_fail += 1
                    continue
                if name == 'Examples' and ch.isDirectory():
                    try:
                        j = File(ch, 'JavaScript')
                        p = File(ch, 'Python')
                        if j.exists() and j.isDirectory():
                            for jc in j.listFiles() or []:
                                try:
                                    _recursive_delete(jc)
                                except Exception:
                                    scripts_fail += 1
                        if p.exists() and p.isDirectory():
                            for pc in p.listFiles() or []:
                                try:
                                    _recursive_delete(pc)
                                except Exception:
                                    scripts_fail += 1
                    except Exception:
                        scripts_fail += 1
                    continue
                try:
                    _recursive_delete(ch)
                except Exception:
                    scripts_fail += 1
            except Exception:
                scripts_fail += 1
        if scripts_fail:
            safe_log("Cleanup", "Failed: scripts cleanup (%s items)" % scripts_fail)
    except Exception:
        safe_log("Cleanup", "Failed: scripts cleanup block")

    try:
        from java.lang import Thread
        Thread.sleep(200)
    except Exception:
        pass

    safe_log("Cleanup", "Completed")
    return v2_dir

def download_file(url, target_dir):

    filename = get_filename_from_url(url)
    target_path = target_dir + os.sep + filename
    target_dir_obj = File(target_dir)

    do_mkdirs_file(target_dir_obj)

    input_stream = None
    output_stream = None
    connection = None

    try:
        url_obj = URL(url)
        try:
            input_stream = BufferedInputStream(url_obj.openStream())
        except Exception:
            try:
                connection = url_obj.openConnection()
                try:
                    connection.setRequestProperty("User-Agent", "Mozilla/5.0")
                except Exception:
                    pass
                try:
                    connection.setConnectTimeout(10000)
                    connection.setReadTimeout(30000)
                except Exception:
                    pass
                input_stream = BufferedInputStream(connection.getInputStream())
            except Exception:
                safe_log("Download", "Failed")
                return False

        output_stream = FileOutputStream(target_path)
        buffer = zeros(4096, 'b')
        bytes_read = input_stream.read(buffer)

        downloaded_bytes = 0
        safe_print("Downloading from www.AlanDesign.org -> %s" % _format_path_for_log(target_path))
        
        
        while bytes_read != -1:
            if bytes_read > 0:
                output_stream.write(buffer, 0, bytes_read)
                downloaded_bytes += bytes_read
            bytes_read = input_stream.read(buffer)

        try:
            output_stream.flush()
        except Exception:
            pass

        #safe_log("Download", "Completed: size %s bytes" % downloaded_bytes)
        safe_print("zip size %s bytes" % downloaded_bytes)

        return True

    except Exception:
        safe_log("Download", "Failed")
        return False

    finally:
        try:
            if input_stream is not None:
                input_stream.close()
        except Exception:
            pass
        try:
            if output_stream is not None:
                output_stream.close()
        except Exception:
            pass

def unzip_file(zip_path, target_dir):

    zin = None
    skipped = 0
    try:
        zf = File(zip_path)
        if not zf.exists():
            safe_log("unzip", "Failed")
            return False

        td = File(target_dir)
        do_mkdirs_file(td)

        safe_log("unzip", "Starting: %s -> %s" % (_format_path_for_log(zip_path), _format_path_for_log(target_dir)))##
        zin = ZipInputStream(FileInputStream(zip_path))
        entry = zin.getNextEntry()
        buffer = zeros(4096, 'b')
        try:
            target_canon = File(target_dir).getCanonicalPath()
        except Exception:
            target_canon = None

        while entry is not None:
            name = entry.getName()
            dest = File(target_dir, name)
            try:
                if target_canon is not None:
                    dest_canon = dest.getCanonicalPath()
                    if not dest_canon.startswith(target_canon + os.sep) and dest_canon != target_canon:
                        skipped += 1
                        zin.closeEntry()
                        entry = zin.getNextEntry()
                        continue
            except Exception:
                pass

            if entry.isDirectory():
                do_mkdirs_file(dest)
            else:
                parent = dest.getParentFile()
                if parent is not None and not parent.exists():
                    do_mkdirs_file(parent)
                out = None
                try:
                    out = BufferedOutputStream(FileOutputStream(dest.getAbsolutePath()))
                    read = zin.read(buffer)
                    while read != -1:
                        if read > 0:
                            out.write(buffer, 0, read)
                        read = zin.read(buffer)
                    try:
                        out.flush()
                    except Exception:
                        pass
                finally:
                    try:
                        if out is not None:
                            out.close()
                    except Exception:
                        pass

            zin.closeEntry()
            entry = zin.getNextEntry()

        try:
            zin.close()
        except Exception:
            pass

        if skipped:
            safe_log("unzip", "ZipSlip skipped: %s" % skipped)
        #safe_log("unzip", "Completed")
        return True

    except Exception:
        safe_log("unzip", "Failed")
        try:
            if zin is not None:
                zin.close()
        except Exception:
            pass
        return False

def reload_and_save_config():
    try:
        from java.lang import ClassLoader, String, Throwable, System
        from java.nio.file import Files as NioFiles, Paths
        from java.util import HashSet
        from java.util.regex import Pattern
        from javax.swing.tree import TreePath
        from javax.swing import SwingUtilities
        from java.lang import Runnable
        from org.openpnp.util import IdentifiableList
        from org.apache.commons.io import FileUtils
        try:
            from org.openpnp.machine.reference.vision import ReferenceBottomVision as _RBVClass
        except Exception:
            _RBVClass = None


        def _get_private_field(obj, field_name):
            cls = obj.getClass()
            while cls is not None:
                try:
                    field = cls.getDeclaredField(field_name)
                    field.setAccessible(True)
                    return field.get(obj)
                except Exception:
                    cls = cls.getSuperclass()
                    continue
                except Throwable:
                    cls = cls.getSuperclass()
            raise AttributeError(field_name)

        def _get_field_by_name(obj, field_name):
            cls = obj.getClass()
            while cls is not None:
                try:
                    for field in cls.getDeclaredFields():
                        if field.getName() == field_name:
                            field.setAccessible(True)
                            return field.get(obj)
                except Exception:
                    pass
                cls = cls.getSuperclass()
            return None

        def _set_field_by_name(obj, field_name, value):
            cls = obj.getClass()
            while cls is not None:
                try:
                    for field in cls.getDeclaredFields():
                        if field.getName() == field_name:
                            field.setAccessible(True)
                            field.set(obj, value)
                            return True
                except Exception:
                    pass
                cls = cls.getSuperclass()
            return False

        def _file_contains_text(file_obj, text):
            if file_obj is None or not file_obj.exists():
                return False
            raw = Paths.get(file_obj.getPath())
            data = NioFiles.readAllBytes(raw)
            haystack = String(data, "UTF-8")
            return haystack.contains(String(text))

        def _get_config_file_or_default(config_dir, filename, resource_name):
            file_obj = File(config_dir, filename)
            if file_obj.exists():
                return file_obj
            if Logger is not None:
                Logger.warn("Python: {} not found, loading defaults.", filename)
            temp_file = File.createTempFile(filename.replace(".", "_"), "xml")
            FileUtils.copyURLToFile(ClassLoader.getSystemResource(resource_name), temp_file)
            return temp_file

        def _extract_alignment_ids(machine_file):
            ids = HashSet()
            if machine_file is None or not machine_file.exists():
                return ids
            raw = Paths.get(machine_file.getPath())
            data = NioFiles.readAllBytes(raw)
            text = String(data, "UTF-8")
            matcher = Pattern.compile("bottom-vision-id=\"([^\"]+)\"").matcher(text)
            while matcher.find():
                ids.add(matcher.group(1))
            return ids

        def _filter_part_alignments_by_file(cfg, machine_file):
            machine = cfg.getMachine()
            part_alignments = list(machine.getPartAlignments())
            allowed_ids = _extract_alignment_ids(machine_file)
            if allowed_ids.isEmpty():
                return
            new_list = IdentifiableList()
            for alignment in part_alignments:
                bottom_vision_id = _get_field_by_name(alignment, "bottomVisionId")
                if bottom_vision_id is not None and allowed_ids.contains(bottom_vision_id):
                    new_list.add(alignment)
            _set_field_by_name(machine, "partAlignments", new_list)

        def _remove_stock_vision_settings_if_missing(cfg, vision_file):
            if _file_contains_text(vision_file, 'id="BVS_Stock"'):
                return
            for settings in list(cfg.getVisionSettings()):
                try:
                    if settings.getId() == "BVS_Stock":
                        try:
                            cfg.removeVisionSettings(settings)
                        except Exception:
                            pass
                except Exception:
                    pass
            try:
                cfg.fireVisionSettingsChanged()
            except Exception:
                pass

        def _notify_configuration(cfg):
            listeners = list(_get_private_field(cfg, "listeners"))
            for listener in listeners:
                try:
                    listener.configurationComplete(cfg)
                except Exception as e:
                    if Logger is not None:
                        Logger.error("Python: configurationComplete listener failed: {}", _safe_error_text(e))

        def _safe_error_text(e):
            try:
                text = str(e)
            except Exception:
                try:
                    text = String(e)
                except Exception:
                    return "<unprintable>"
            try:
                return text.encode("ascii", "backslashreplace").decode("ascii")
            except Exception:
                return "<unprintable>"

        def _log_cause(e):
            try:
                cause = e.getCause()
                if cause is not None:
                    if Logger is not None:
                        Logger.error("Python: cause: {}", cause)
            except Exception:
                pass

        def _run_step(name, fn):
            try:
                fn()
                return True
            except Throwable as e:
                if Logger is not None:
                    Logger.error("Python: step {} FAILED (Throwable): {}", name, _safe_error_text(e))
                _log_cause(e)
                return False
            except BaseException as e:
                if Logger is not None:
                    Logger.error("Python: step {} FAILED: {}", name, _safe_error_text(e))
                _log_cause(e)
                return False

        def reload_machine_and_vision_settings():
            cfg = Configuration.get()
            config_dir = cfg.getConfigurationDirectory()
            serializer = Configuration.createSerializer()

            machine_file = File(config_dir, "machine.xml")
            vision_file = File(config_dir, "vision-settings.xml")
            packages_file = File(config_dir, "packages.xml")
            parts_file = File(config_dir, "parts.xml")
            boards_file = File(config_dir, "boards.xml")
            panels_file = File(config_dir, "panels.xml")
            script_state_file = File(config_dir, "script-state.xml")

            def _load_machine():
                if Logger is not None:
                    Logger.info("Python: reloading machine.xml without cfg.load()...")
                machine_holder = serializer.read(Configuration.MachineConfigurationHolder, machine_file)
                machine = _get_private_field(machine_holder, "machine")
                cfg.setMachine(machine)
            if not _run_step("load_machine", _load_machine):
                return False

            def _load_vision():
                if Logger is not None:
                    Logger.info("Python: reloading vision-settings.xml without cfg.load()...")
                vision_holder = serializer.read(Configuration.VisionSettingsConfigurationHolder, vision_file)
                for existing in list(cfg.getVisionSettings()):
                    try:
                        cfg.removeVisionSettings(existing)
                    except Exception:
                        pass
                if vision_holder is not None and vision_holder.visionSettings is not None:
                    for settings in vision_holder.visionSettings:
                        try:
                            cfg.addVisionSettings(settings)
                        except Exception:
                            pass
            if not _run_step("load_vision_settings", _load_vision):
                return False

            def _load_packages():
                if Logger is not None:
                    Logger.info("Python: reloading packages.xml without cfg.load()...")
                packages_src = _get_config_file_or_default(config_dir, "packages.xml", "config/packages.xml")
                packages_holder = serializer.read(Configuration.PackagesConfigurationHolder, packages_src)
                for existing in list(cfg.getPackages()):
                    try:
                        cfg.removePackage(existing)
                    except Exception:
                        pass
                packages = _get_private_field(packages_holder, "packages")
                if packages is not None:
                    for pkg in list(packages):
                        try:
                            cfg.addPackage(pkg)
                        except Exception:
                            pass
            if not _run_step("load_packages", _load_packages):
                return False

            def _load_parts():
                if Logger is not None:
                    Logger.info("Python: reloading parts.xml without cfg.load()...")
                parts_src = _get_config_file_or_default(config_dir, "parts.xml", "config/parts.xml")
                parts_holder = serializer.read(Configuration.PartsConfigurationHolder, parts_src)
                for existing in list(cfg.getParts()):
                    try:
                        cfg.removePart(existing)
                    except Exception:
                        pass
                parts = _get_private_field(parts_holder, "parts")
                if parts is not None:
                    for part in list(parts):
                        try:
                            cfg.addPart(part)
                        except Exception:
                            pass
            if not _run_step("load_parts", _load_parts):
                return False

            def _link_part_packages():
                for part in list(cfg.getParts()):
                    try:
                        if part.getPackage() is not None:
                            continue
                    except Exception:
                        pass
                    try:
                        package_id = _get_private_field(part, "packageId")
                    except Exception:
                        package_id = None
                    if package_id is None:
                        continue
                    try:
                        pkg = cfg.getPackage(package_id)
                    except Exception:
                        pkg = None
                    if pkg is None:
                        if Logger is not None:
                            Logger.warn("Python: package not found for part (packageId={})", package_id)
                        continue
                    try:
                        part.setPackage(pkg)
                    except Exception:
                        pass
            if not _run_step("link_part_packages", _link_part_packages):
                return False

            def _load_boards():
                if Logger is not None:
                    Logger.info("Python: reloading boards.xml without cfg.load()...")
                boards_src = _get_config_file_or_default(config_dir, "boards.xml", "config/boards.xml")
                boards_holder = serializer.read(Configuration.BoardsConfigurationHolder, boards_src)
                for existing in list(cfg.getBoards()):
                    try:
                        cfg.removeBoard(existing)
                    except Exception:
                        pass
                boards = _get_private_field(boards_holder, "boards")
                if boards is not None:
                    for board_file in list(boards):
                        try:
                            cfg.addBoard(board_file)
                        except Exception:
                            pass
            if not _run_step("load_boards", _load_boards):
                return False

            def _load_panels():
                if Logger is not None:
                    Logger.info("Python: reloading panels.xml without cfg.load()...")
                panels_src = _get_config_file_or_default(config_dir, "panels.xml", "config/panels.xml")
                panels_holder = serializer.read(Configuration.PanelsConfigurationHolder, panels_src)
                for existing in list(cfg.getPanels()):
                    try:
                        cfg.removePanel(existing)
                    except Exception:
                        pass
                panels = _get_private_field(panels_holder, "panels")
                if panels is not None:
                    for panel_file in list(panels):
                        try:
                            cfg.addPanel(panel_file)
                        except Exception:
                            pass
            if not _run_step("load_panels", _load_panels):
                return False

            def _load_script_state():
                if Logger is not None:
                    Logger.info("Python: reloading script-state.xml without cfg.load()...")
                state_src = _get_config_file_or_default(config_dir, "script-state.xml", "config/script-state.xml")
                state_holder = serializer.read(Configuration.ScriptStateConfigurationHolder, state_src)
                state = _get_private_field(state_holder, "scriptState")
                if state is not None:
                    cfg.scriptState = state
            if not _run_step("load_script_state", _load_script_state):
                return False

            if not _run_step("filter_part_alignments", lambda: _filter_part_alignments_by_file(cfg, machine_file)):
                return False
            if not _run_step("notify_configuration", lambda: _notify_configuration(cfg)):
                return False
            if not _run_step("remove_stock_settings", lambda: _remove_stock_vision_settings_if_missing(cfg, vision_file)):
                return False
            if Logger is not None:
                Logger.info("Python: reload finished.")
            return True

        def _log_info(msg):
            try:
                if Logger is not None:
                    Logger.info(msg)
            except Exception:
                pass
            try:
                System.out.println(str(msg))
            except Exception:
                pass

        def _log_error(msg):
            try:
                if Logger is not None:
                    Logger.error(msg)
            except Exception:
                pass
            try:
                System.err.println(str(msg))
            except Exception:
                try:
                    System.out.println(str(msg))
                except Exception:
                    pass

        def find_vision_setting_by_display_name(name):
            try:
                cfg = Configuration.get()
                candidates = []
                for vs in list(cfg.getVisionSettings()):
                    try:
                        if vs is None:
                            continue
                        vname = None
                        try:
                            vname = vs.getName()
                        except Exception:
                            try:
                                vname = str(vs)
                            except Exception:
                                vname = None
                        if vname == name:
                            return vs
                        if vname is not None:
                            candidates.append((vname, vs))
                    except Exception:
                        continue
                lname = (name or '').lower()
                for vname, vs in candidates:
                    try:
                        if vname is None:
                            continue
                        if lname in vname.lower():
                            _log_info('Fuzzy-matched vision setting "%s" for requested "%s"' % (vname, name))
                            return vs
                    except Exception:
                        continue
            except Exception as e:
                _log_error('Error enumerating vision settings: %s' % str(e))
            return None

        def find_reference_bottom_vision(machine):
            try:
                if machine is None:
                    return None
                try:
                    pals = list(machine.getPartAlignments())
                except Exception:
                    pals = []
                for p in pals:
                    try:
                        if _RBVClass is not None and isinstance(p, _RBVClass):
                            return p
                    except Exception:
                        pass
                    try:
                        cname = p.getClass().getName()
                    except Exception:
                        cname = ''
                    if 'ReferenceBottomVision' in cname:
                        return p
                    try:
                        s = str(p)
                    except Exception:
                        s = ''
                    if 'ReferenceBottomVision' in s:
                        return p
            except Exception:
                pass
            return None

        def find_reference_fiducial_locator(machine):
            try:
                if machine is None:
                    return None
                try:
                    locator = machine.getFiducialLocator()
                    if locator is not None:
                        return locator
                except Exception:
                    pass
                for p in list(machine.getPartAlignments()):
                    try:
                        cname = p.getClass().getName()
                    except Exception:
                        cname = str(p)
                    if 'ReferenceFiducialLocator' in cname:
                        return p
                    try:
                        s = str(p)
                    except Exception:
                        s = ''
                    if 'ReferenceFiducialLocator' in s:
                        return p
            except Exception:
                pass
            return None

        def select_in_machine_setup(targetObj):
            try:
                frame = MainFrame.get()
                if frame is None:
                    return False
                panel = frame.getMachineSetupTab()
                if panel is None:
                    return False
                try:
                    fld = panel.getClass().getDeclaredField('tree')
                    fld.setAccessible(True)
                    tree = fld.get(panel)
                except Exception:
                    return False
                if tree is None:
                    return False
                model = tree.getModel()
                root = model.getRoot()
                try:
                    target_class_name = targetObj.getClass().getName()
                except Exception:
                    target_class_name = None
                try:
                    target_str = str(targetObj)
                except Exception:
                    target_str = None

                NodeClass = None
                try:
                    NodeClass = root.getClass()
                except Exception:
                    NodeClass = None
                objField = None
                childrenField = None
                if NodeClass is not None:
                    try:
                        objField = NodeClass.getDeclaredField('obj')
                        objField.setAccessible(True)
                    except Exception:
                        objField = None
                    try:
                        childrenField = NodeClass.getDeclaredField('children')
                        childrenField.setAccessible(True)
                    except Exception:
                        childrenField = None
                found_path_container = [None]

                def dfs(node, stack):
                    if found_path_container[0] is not None:
                        return
                    stack.append(node)
                    try:
                        nodeObj = None
                        if objField is not None:
                            try:
                                nodeObj = objField.get(node)
                            except Exception:
                                nodeObj = None
                        else:
                            try:
                                nodeObj = node.obj
                            except Exception:
                                nodeObj = None
                        if nodeObj is targetObj:
                            found_path_container[0] = list(stack)
                            stack.pop()
                            return
                        try:
                            if nodeObj is not None:
                                try:
                                    node_class_name = nodeObj.getClass().getName()
                                except Exception:
                                    node_class_name = None
                                try:
                                    node_str = str(nodeObj)
                                except Exception:
                                    node_str = None
                                if target_class_name is not None and node_class_name is not None and target_class_name == node_class_name:
                                    found_path_container[0] = list(stack)
                                    stack.pop()
                                    return
                                if target_str is not None and node_str is not None and target_str == node_str:
                                    found_path_container[0] = list(stack)
                                    stack.pop()
                                    return
                        except Exception:
                            pass
                    except Exception:
                        pass
                    try:
                        children = None
                        if childrenField is not None:
                            try:
                                children = childrenField.get(node)
                            except Exception:
                                children = None
                        if children is None:
                            try:
                                it = node.children()
                                children = []
                                while it.hasMoreElements():
                                    children.append(it.nextElement())
                            except Exception:
                                children = None
                        if children is not None:
                            for c in list(children):
                                dfs(c, stack)
                                if found_path_container[0] is not None:
                                    break
                    except Exception:
                        pass
                    try:
                        stack.pop()
                    except Exception:
                        pass
                try:
                    dfs(root, [])
                except Exception:
                    pass
                if found_path_container[0] is None:
                    return False
                try:
                    arr = found_path_container[0]
                    tp = TreePath(arr[0])
                    if len(arr) > 1:
                        tp = TreePath(arr)
                    try:
                        class _Sel(Runnable):
                            def run(self):
                                try:
                                    tree.setSelectionPath(tp)
                                except Exception:
                                    pass
                        SwingUtilities.invokeLater(_Sel())
                    except Exception:
                        try:
                            tree.setSelectionPath(tp)
                        except Exception:
                            return False
                    return True
                except Exception:
                    return False
            except Exception:
                return False

        def selete_main():
            TARGET_NAME = "- TornadoSMT V2 Bottom Vision -"
            FIDUCIAL_TARGET_NAME = "- TornadoSMT V2 Fiducial Locator -"
            _log_info('Start Vision Setting')#6
            

            try:
                cfg_deadline = time.time() + 5.0
                while time.time() < cfg_deadline:
                    try:
                        if Configuration.isInstanceInitialized():
                            break
                    except Exception:
                        pass
                    _log_info('Waiting for Configuration to initialize...')
                    time.sleep(0.2)
                if not Configuration.isInstanceInitialized():
                    _log_error('Configuration not initialized; aborting script')
                    return
            except Exception:
                pass
            cfg = Configuration.get()
            try:
                frame_deadline = time.time() + 5.0
                while time.time() < frame_deadline:
                    try:
                        frame = MainFrame.get()
                        if frame is not None:
                            break
                    except Exception:
                        pass
                    _log_info('Waiting for MainFrame to initialize...')
                    time.sleep(0.2)
            except Exception:
                pass
            machine = cfg.getMachine()
            if machine is None:
                _log_error('No machine configured (Configuration.get().getMachine() returned None)')
                return
            rbv = find_reference_bottom_vision(machine)
            if rbv is None:
                _log_error('ReferenceBottomVision instance not found')
                return
            vs = find_vision_setting_by_display_name(TARGET_NAME)
            if vs is None:
                _log_error('Vision setting with name "%s" not found' % TARGET_NAME)
                return
            try:
                rbv.setBottomVisionSettings(vs)
                _log_info('rbv.setBottomVisionSettings(...) invoked')
            except Exception:
                traceback.print_exc()
                return
            fvs = find_vision_setting_by_display_name(FIDUCIAL_TARGET_NAME)
            if fvs is not None:
                fid = find_reference_fiducial_locator(machine)
                if fid is not None:
                    try:
                        fid.setFiducialVisionSettings(fvs)
                        _log_info('fid.setFiducialVisionSettings(...) invoked')
                    except Exception:
                        pass
            else:
                _log_error('Fiducial vision setting with name "%s" not found' % FIDUCIAL_TARGET_NAME)
            try:
                frame = MainFrame.get()
                if frame is not None:
                    _log_info('Calling MainFrame.saveConfig() to persist changes...')
                    ok = frame.saveConfig()
                    if ok:
                        _log_info('Configuration saved successfully.')
                    else:
                        _log_error('MainFrame.saveConfig() returned False')
                else:
                    _log_error('MainFrame.get() returned None; not saving automatically')
            except Exception as e:
                _log_error('Error calling MainFrame.saveConfig(): %s' % str(e))
                traceback.print_exc()
            try:
                frame = MainFrame.get()
                if frame is not None and frame.getMachineSetupTab() is not None:
                    _log_info('Refreshing Machine Setup tree selection')
                    frame.getMachineSetupTab().selectCurrentTreePath()
            except Exception:
                pass
            try:
                select_in_machine_setup(rbv)
            except Exception:
                pass
            _log_info('Vision Setting completed')#6

        if Configuration is None:
            safe_print('Configuration API not available in this environment; skipping reload/init.')
            return False
        if not reload_machine_and_vision_settings():
            return False
        time.sleep(WAIT_SECONDS)
        selete_main()
        return True

    except Exception as e:
        try:
            if Logger is not None:
                Logger.error('Python: reload/init FAILED: {}', e)
            else:
                safe_print('reload/init FAILED: %s' % str(e))
            traceback.print_exc()
        except Exception:
            pass
        return False


# ===========================================
# 备用拷贝：主复制失败时的回退方案
# ===========================================
def _copy_result(ok, reason=None):
    if ok:
        #safe_log("Copy", "Completed")
        return
    if reason:
        safe_log("Copy", "Failed: %s" % reason)
    else:
        safe_log("Copy", "Failed")


def fallback_copy_file(src_file, dest_file):
    try:
        in_s = None
        out_s = None
        try:
            in_s = FileInputStream(src_file)
            parent = dest_file.getParentFile()
            if parent is not None and not parent.exists():
                parent.mkdirs()
            out_s = FileOutputStream(dest_file)
            buf = zeros(4096, 'b')
            read = in_s.read(buf)
            while read != -1:
                if read > 0:
                    out_s.write(buf, 0, read)
                read = in_s.read(buf)
            try:
                out_s.flush()
            except Exception:
                pass
            return True
        finally:
            try:
                if in_s is not None:
                    in_s.close()
            except Exception:
                pass
            try:
                if out_s is not None:
                    out_s.close()
            except Exception:
                pass
    except Exception:
        return False

def copy_with_verify(src, dest):
    #safe_log("Copy", "Starting")
    try:
        if src.isDirectory():
            _recursive_copy(src, dest)
            _copy_result(True)
            return True
        _recursive_copy(src, dest)
        try:
            src_len = src.length()
            dest_len = dest.length()
            if dest_len == src_len and dest_len > 0:
                _copy_result(True)
                return True
        except Exception:
            pass
        try:
            if dest.exists():
                _recursive_delete(dest)
        except Exception:
            pass
        ok = fallback_copy_file(src, dest)
        if not ok:
            _copy_result(False, "fallback copy failed")
            return False
        try:
            if dest.exists() and dest.length() == src.length() and dest.length() > 0:
                _copy_result(True)
                return True
            _copy_result(False, "verification failed after fallback")
            return False
        except Exception:
            _copy_result(False, "verification failed after fallback")
            return False
    except Exception:
        _copy_result(False, "exception")
        return False

def _config_has_var(config_path):
    if not config_path or not os.path.exists(config_path):
        return False
    try:
        try:
            with io.open(config_path, "r", encoding="utf-8", errors="ignore") as f:
                content = f.read()
        except Exception:
            with io.open(config_path, "r") as f:
                content = f.read()
        pattern = r"^" + re.escape(VAR_NAME) + r"\s+\S+"
        return re.search(pattern, content, re.MULTILINE) is not None
    except Exception:
        return False


def find_usb_with_ctypes():
    try:
        import ctypes
    except Exception:
        return None
    candidates = []
    try:
        for letter in "ABCDEFGHIJKLMNOPQRSTUVWXYZ":
            root = letter + ":\\"
            if not os.path.exists(root):
                continue
            try:
                if ctypes.windll.kernel32.GetDriveTypeW(root) != 2:
                    continue
            except Exception:
                continue
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(root)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测（Java NIO）
# 条件：config.txt 中包含目标变量
# ===========================================
def find_usb_with_java():
    try:
        import java.io as jio
    except Exception:
        return None
    candidates = []
    try:
        roots = jio.File.listRoots()
        for r in roots:
            root = r.getAbsolutePath()
            cfg = os.path.join(root, "config.txt")
            if os.path.exists(cfg) and _config_has_var(cfg):
                candidates.append(root)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测（macOS /Volumes）
# 条件：config.txt 中包含目标变量
# ===========================================
def find_usb_on_macos():
    try:
        vols_root = '/Volumes'
        if not os.path.exists(vols_root):
            return None
    except Exception:
        return None
    candidates = []
    try:
        for name in os.listdir(vols_root):
            path = os.path.join(vols_root, name)
            cfg = os.path.join(path, 'config.txt')
            if os.path.exists(cfg) and _config_has_var(cfg):
                root = path if path.endswith(os.sep) else path + os.sep
                candidates.append(root)
    except Exception:
        return None
    return candidates or None


# ===========================================
# USB 检测入口：按平台选择检测方式
# ===========================================
def find_usb():
    global _USB_FIND_STATUS
    _USB_FIND_STATUS = None
    candidates = []
    try:
        r = find_usb_with_ctypes()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass
    try:
        from java.lang import System
        osname = System.getProperty('os.name')
    except Exception:
        osname = sys.platform
    if osname and 'mac' in osname.lower():
        try:
            r = find_usb_on_macos()
            if r:
                candidates.extend(list(r))
        except Exception:
            pass
    try:
        r = find_usb_with_java()
        if r:
            candidates.extend(list(r))
    except Exception:
        pass

    unique = []
    seen = set()
    for p in candidates:
        try:
            key = p.lower()
        except Exception:
            key = p
        if key not in seen:
            seen.add(key)
            unique.append(p)

    if not unique:
        _USB_FIND_STATUS = "none"
        return None
    if len(unique) > 1:
        _USB_FIND_STATUS = "multiple"
        try:
            if _LANG == "en":
                msg = u"Multiple USB drives with config.txt found. Please check."
                title = u"Multiple USB Found"
            else:
                msg = u"找到多个含有config.txt的U盘，请检查。"
                title = u"找到多个U盘"
            JOptionPane.showMessageDialog(_PROGRESS_DIALOG.dialog if _PROGRESS_DIALOG else None, msg, title, JOptionPane.WARNING_MESSAGE)
        except Exception:
            pass
        return None
    _USB_FIND_STATUS = "ok"
    return unique[0]


# ===========================================
# 同步 config.txt：备份并写回 USB
# ===========================================
def sync_config_from_usb():
    try:
        usb_root = None
        try:
            usb_root = _USB_SELECTED_ROOT
        except Exception:
            usb_root = None
        if not usb_root:
            if _USB_FIND_STATUS == "multiple":
                return
            safe_print("No matching USB drive found with config.txt. Operation stopped.")
            return

        usb_cfg = os.path.join(usb_root, "config.txt")
        if not os.path.exists(usb_cfg):
            safe_print("config.txt not found on USB drive. Operation stopped.")
            return


        if not USB_CONFIG_DIR:
            safe_print("USB_CONFIG_DIR is not set; cannot back up config.txt")
            return
        backup_cfg = os.path.join(USB_CONFIG_DIR, "config.txt")
        try:
            shutil.copyfile(usb_cfg, backup_cfg)
        except Exception as e:
            safe_print("Backup failed: %s" % str(e))
            return

        source_path = backup_cfg

        target_dir_local = TARGET_DIR
        target_path = os.path.join(target_dir_local, "config.txt")
        if not os.path.exists(target_path):
            safe_print("Missing target config.txt: %s" % target_path)
            return

        with io.open(source_path, "r", encoding="utf-8") as f:
            a_content = f.read()
        m = re.search(r"^" + re.escape(VAR_NAME) + r"\s+(\S+)", a_content, re.MULTILINE)
        if not m:
            safe_print("Variable %s not found in source config.txt." % VAR_NAME)
            return
        value = m.group(1)

        with io.open(target_path, "r", encoding="utf-8") as f:
            b_content = f.read()
        pattern = r"^(" + re.escape(VAR_NAME) + r")(\s+)(\S+)(.*)$"
        def _repl_var(m):
            return m.group(1) + m.group(2) + value + m.group(4)
        new = re.sub(pattern, _repl_var, b_content, count=1, flags=re.MULTILINE)

        def _parse_version_from_text(text):
            try:
                mver = re.search(r"#Version:V(\d+(?:\.\d+)?)", text)
                if not mver:
                    return None
                return float(mver.group(1))
            except Exception:
                return None

        src_version = _parse_version_from_text(a_content)
        if src_version is None or src_version < 2.5:
            new_before = new
            new = re.sub(r"#Version:V(\d+(?:\.\d+)?)", "#Version:V2.1", new, count=1)
            def _repl_play(m):
                return m.group(1) + "4.28!"
            new = re.sub(r"(play_led_pin\s+)4\.28!*", _repl_play, new, count=1)
            if new != new_before:
                safe_print("Done: version and play_led_pin modifications completed.")#7

        with io.open(target_path, "w", encoding="utf-8") as f:
            f.write(new)

        safe_print("Done: backed up USB drive config.txt to: %s" % _format_path_for_log(backup_cfg))
        safe_print("Done: transferred '%s' value %s to target %s" % (VAR_NAME, value, _format_path_for_log(target_path)))
        

        try:
            dest_cfg = os.path.join(usb_root, "config.txt")
            try:
                shutil.copyfile(target_path, dest_cfg)
                safe_print("Done: copy target config.txt to USB drive: %s" % dest_cfg)#7
            except Exception as e:
                safe_print("Failed to copy target config.txt to USB drive: %s" % str(e))
                return

            safe_print("Done: config.txt synced to USB drive successfully (only config.txt updated; other USB files preserved).")#7
            safe_log("USB Sync", "Completed")#7
            try:
                try:
                    _delete_self_script()
                except Exception:
                    pass
                try:
                    openpnp_root = get_openpnp_root()
                    v2_dir = os.path.join(openpnp_root, 'TornadoSMT_V2')
                    v2_file = File(v2_dir)
                    if v2_file.exists():
                        _recursive_delete(v2_file)
                        safe_print("Delete TornadoSMT_V2 folder: %s" % v2_dir)#8
                except Exception:
                    pass
                try:
                    safe_print("Waiting %s seconds..." % RESULT_DELAY_SECONDS)
                    time.sleep(float(RESULT_DELAY_SECONDS))
                except Exception:
                    pass
                _progress_show_final()
            except Exception:
                pass
            return
        except Exception as e:
            safe_print("USB drive sync operation error: %s" % str(e))

    except Exception as e:
        safe_print("Error in sync_config_from_usb: %s" % str(e))


# ===========================================
# 主流程：串联下载、合并、重载与同步
# ===========================================
def _run_pipeline():
    global TARGET_DIR
    safe_log("TornadoSMT_V2", "Deployment Starting")#1
    try:
        _progress_set_i18n(5, "progress.preparing", u"Preparing...", u"Preparing...")
        TARGET_DIR = resolve_target_dir()
    except Exception:
        TARGET_DIR = DEFAULT_TARGET_DIR
        safe_log("TornadoSMT_V2", "Prepare: Failed")

    safe_log("Download", "Starting")#2
    _progress_set_i18n(15, "progress.downloading", u"Downloading...", u"Downloading...")

    _success = download_file(DOWNLOAD_URL, TARGET_DIR)
    if not _success:
        safe_log("TornadoSMT_V2", "Download: Failed")
        try:
            if Desktop.isDesktopSupported():
                Desktop.getDesktop().browse(URI(DOWNLOAD_URL))
        except Exception:
            pass
    else:
        safe_log("Download", "Completed") 
        try:
            zip_path = TARGET_DIR + os.sep + get_filename_from_url(DOWNLOAD_URL)
            safe_log("Unzip", "Starting")#3
            _progress_set_i18n(35, "progress.unzipping", u"Unzipping...", u"Unzipping...")
            _unz = unzip_file(zip_path, TARGET_DIR)
            if _unz:
                safe_log("Unzip", "Completed")#3
                _progress_set_i18n(45, "progress.unzip_done", u"Unzip done", u"Unzip done")
                try:
                    f = File(zip_path)
                    try:
                        f.delete()
                    except Exception:
                        pass
                except Exception:
                    pass
            else:
                safe_log("TornadoSMT_V2", "Unzip: Failed")
        except Exception:
            safe_log("TornadoSMT_V2", "Unzip: Failed")

    try:
        openpnp_root = get_openpnp_root()
        source_dot_openpnp = os.path.join(TARGET_DIR, '.openpnp2')
        src_dir = File(source_dot_openpnp)
        if src_dir.exists() and src_dir.isDirectory():
            safe_log("Merge", "Starting")#4
            _progress_set_i18n(60, "progress.merging", u"Merging config...", u"Merging config...")
            for item in src_dir.listFiles() or []:
                try:
                    if item is None:
                        continue
                    name = item.getName()
                    dest = File(openpnp_root, name)
                    if dest.exists():
                        if not (item.isDirectory() and dest.isDirectory()):
                            try:
                                _recursive_delete(dest)
                            except Exception:
                                pass
                    safe_print("Merge %s -> %s" % (_format_java_file_path_for_log(item), _format_java_file_path_for_log(dest)))##
                    try:
                        copy_with_verify(item, dest)
                    except Exception:
                        pass
                except Exception:
                    pass
            safe_log("Merge", "Completed")#4
        else:
            safe_log("TornadoSMT_V2", "Merge: Failed")
    except Exception:
        safe_log("TornadoSMT_V2", "Merge: Failed")

    try:
        safe_log("Reload and Save", "Starting")#5
        _progress_set_i18n(80, "progress.reloading", u"Reloading config...", u"Reloading config...")
        ok = reload_and_save_config()
        if ok:
            safe_log("Reload and saveConfig", "Completed")#5
            _progress_set_i18n(90, "progress.syncing", u"Syncing USB...", u"Syncing USB...")
            try:
                safe_log("USB Sync", "Starting")#7
                sync_config_from_usb()
                #safe_log("TornadoSMT_V2", "USB Sync: Completed")
            except Exception:
                safe_log("TornadoSMT_V2", "USB Sync: Failed")
        else:
            safe_log("TornadoSMT_V2", "Reload and saveConfig: Failed")
    except Exception:
        safe_log("TornadoSMT_V2", "Reload and saveConfig: Failed")

    _progress_set_i18n(100, "progress.done", u"Done", u"Done")

def _start_pipeline_async():
    try:
        from java.lang import Thread
    except Exception:
        _run_pipeline()
        return

    class _Runner(java.lang.Runnable):
        def run(self):
            _run_pipeline()

    Thread(_Runner()).start()


# ===========================================
# 入口：初始化窗口并启动流程
# ===========================================
def _run_pipeline_async():
    _progress_init()

_run_pipeline_async()

